<?php
/************************************************************************
 * This file is part of EspoCRM.
 *
 * EspoCRM - Open Source CRM application.
 * Copyright (C) 2014-2021 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
 * Website: https://www.espocrm.com
 *
 * EspoCRM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EspoCRM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EspoCRM. If not, see http://www.gnu.org/licenses/.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
 ************************************************************************/

namespace Espo\Tools\Pdf\Tcpdf;

use Espo\Core\Utils\Config;
use Espo\Core\Htmlizer\TemplateRendererFactory;
use Espo\Core\Htmlizer\TemplateRenderer;

use Espo\ORM\Entity;

use Espo\Tools\Pdf\Template;
use Espo\Tools\Pdf\Data;
use Espo\Tools\Pdf\Params;
use Espo\Tools\Pdf\Tcpdf\Tcpdf;

class EntityProcessor
{
    private $fontFace = 'freesans';

    private $fontSize = 12;

    private $config;

    private $templateRendererFactory;

    public function __construct(Config $config, TemplateRendererFactory $templateRendererFactory)
    {
        $this->config = $config;
        $this->templateRendererFactory = $templateRendererFactory;
    }

    public function process(Tcpdf $pdf, Template $template, Entity $entity, Params $params, Data $data): void
    {
        $renderer = $this->templateRendererFactory
            ->create()
            ->setApplyAcl($params->applyAcl())
            ->setEntity($entity)
            ->setData($data->getAdditionalTemplateData());

        $fontFace = $this->config->get('pdfFontFace', $this->fontFace);
        $fontSize = $this->config->get('pdfFontSize', $this->fontSize);

        if ($template->getFontFace()) {
            $fontFace = $template->getFontFace();
        }

        if ($template->hasTitle()) {
            $title = $this->replacePlaceholders($template->getTitle(), $entity);

            $pdf->SetTitle($title);
        }


        $pdf->setFont($fontFace, '', $fontSize, '', true);

        $pdf->setAutoPageBreak(true, $template->getBottomMargin());

        $pdf->setMargins(
            $template->getLeftMargin(),
            $template->getTopMargin(),
            $template->getRightMargin()
        );

        $pageOrientation = $template->getPageOrientation();

        $pageFormat = $template->getPageFormat();

        if ($pageFormat === 'Custom') {
            $pageFormat = [
                $template->getPageWidth(),
                $template->getPageHeight(),
            ];
        }

        $pageOrientationCode = 'P';

        if ($pageOrientation === 'Landscape') {
            $pageOrientationCode = 'L';
        }

        if ($template->hasFooter()) {
            $htmlFooter = $this->render($renderer, $template->getFooter());

            $pdf->setFooterFont([$fontFace, '', $this->fontSize]);
            $pdf->setFooterPosition($template->getFooterPosition());

            $pdf->setFooterHtml($htmlFooter);
        }
        else {
            $pdf->setPrintFooter(false);
        }

        if ($template->hasHeader()) {
            $htmlHeader = $this->render($renderer, $template->getHeader());

            $pdf->setHeaderFont([$fontFace, '', $this->fontSize]);
            $pdf->setHeaderPosition($template->getHeaderPosition());

            $pdf->setHeaderHtml($htmlHeader);
        }
        else {
            $pdf->setPrintHeader(false);
        }

        $pdf->addPage($pageOrientationCode, $pageFormat);

        $htmlBody = $this->render($renderer, $template->getBody());

        $pdf->writeHTML($htmlBody, true, false, true, false, '');

    }

    private function render(TemplateRenderer $renderer, string $template): string
    {
        $html = $renderer->renderTemplate($template);

        return preg_replace_callback(
            '/<barcodeimage data="([^"]+)"\/>/',
            function ($matches) {
                $dataString = $matches[1];

                $data = json_decode(urldecode($dataString), true);

                return $this->composeBarcodeTag($data);
            },
            $html
        );
    }

    private function composeBarcodeTag(array $data): string
    {
        $value = $data['value'] ?? null;

        $codeType = $data['type'] ?? 'CODE128';

        $typeMap = [
            "CODE128" => 'C128',
            "CODE128A" => 'C128A',
            "CODE128B" => 'C128B',
            "CODE128C" => 'C128C',
            "EAN13" => 'EAN13',
            "EAN8" => 'EAN8',
            "EAN5" => 'EAN5',
            "EAN2" => 'EAN2',
            "UPC" => 'UPCA',
            "UPCE" => 'UPCE',
            "ITF14" => 'I25',
            "pharmacode" => 'PHARMA',
            "QRcode" => 'QRCODE,H',
        ];

        switch ($codeType) {
            case 'QRcode':
                $function = 'write2DBarcode';

                $params = [
                    $value,
                    $typeMap[$codeType] ?? null,
                    '', '',
                    $data['width'] ?? 40,
                    $data['height'] ?? 40,
                    [
                        'border' => false,
                        'vpadding' => $data['padding'] ?? 2,
                        'hpadding' => $data['padding'] ?? 2,
                        'fgcolor' => $data['color'] ?? [0, 0, 0],
                        'bgcolor' => $data['bgcolor'] ?? false,
                        'module_width' => 1,
                        'module_height' => 1,
                    ],
                    'N',
                ];
                break;
            case 'SVG' :
                $svg = '@' .file_get_contents('data/upload/618274957c3fe60df');
                $function = 'ImageSVG';
                $params = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1" width="1080" height="1080" viewBox="0 0 1080 1080" xml:space="preserve">
<desc>Created with Fabric.js 4.2.0</desc>
<defs>
</defs>
<g transform="matrix(1 0 0 1 540 540)" id="19aefbe4-d8a1-49f6-990d-3a98ee5925af"  >
<rect style="stroke: none; stroke-width: 1; stroke-dasharray: none; stroke-linecap: butt; stroke-dashoffset: 0; stroke-linejoin: miter; stroke-miterlimit: 4; fill: rgb(255,255,255); fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  x="-540" y="-540" rx="0" ry="0" width="1080" height="1080" />
</g>
<g transform="matrix(Infinity NaN NaN Infinity 0 0)" id="9aee5e25-2949-40dc-b4d2-8baa10c48ba9"  >
</g>
<g transform="matrix(1 0 0 1 249.68 621.52)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-249.68, -621.52)" d="M 249.6548874598071 621.5209003215434 L 249.7148874598071 621.5209003215434" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 555.28 638.19)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-555.28, -638.19)" d="M 255.2411575562701 627.1071704180065 Q 255.2411575562701 627.0771704180065 255.2411575562701 625.6881028938908 Q 255.2411575562701 624.299035369775 255.2411575562701 622.9099678456591 Q 255.2411575562701 621.5209003215434 255.2411575562701 618.742765273312 Q 255.2411575562701 615.9646302250804 255.2411575562701 611.7974276527332 Q 255.2411575562701 607.6302250803859 255.2411575562701 602.0739549839229 Q 255.2411575562701 596.5176848874598 255.2411575562701 589.572347266881 Q 255.2411575562701 582.6270096463023 255.2411575562701 577.0707395498393 Q 255.2411575562701 571.5144694533763 255.2411575562701 564.5691318327974 Q 255.2411575562701 557.6237942122187 255.2411575562701 552.0675241157558 Q 255.2411575562701 546.5112540192927 255.2411575562701 539.5659163987139 Q 255.2411575562701 532.620578778135 255.2411575562701 527.0643086816721 Q 255.2411575562701 521.5080385852091 255.2411575562701 518.7299035369775 Q 255.2411575562701 515.951768488746 255.2411575562701 510.395498392283 Q 255.2411575562701 504.83922829581996 255.2411575562701 500.6720257234727 Q 255.2411575562701 496.50482315112544 255.2411575562701 492.33762057877817 Q 255.2411575562701 488.1704180064309 256.63022508038586 484.00321543408364 Q 258.01929260450163 479.8360128617364 259.40836012861735 475.6688102893891 Q 260.7974276527331 471.50160771704185 263.5755627009646 468.72347266881036 Q 266.35369774919616 465.9453376205788 269.1318327974277 461.77813504823155 Q 271.9099678456592 457.6109324758843 277.46623794212223 452.05466237942125 Q 283.0225080385852 446.4983922829582 288.57877813504825 442.33118971061094 Q 294.1350482315113 438.1639871382637 317.74919614147916 421.4951768488746 Q 341.363344051447 404.82636655948556 366.3665594855306 390.935691318328 Q 391.3697749196142 377.04501607717043 419.15112540192933 364.54340836012864 Q 446.9324758842444 352.04180064308684 478.88102893890675 340.92926045016077 Q 510.82958199356915 329.81672025723475 534.443729903537 325.6495176848875 Q 558.0578778135049 321.4823151125402 584.4501607717043 321.4823151125402 Q 610.8424437299036 321.4823151125402 635.8456591639872 321.4823151125402 Q 660.8488745980708 321.4823151125402 683.0739549839229 327.03858520900326 Q 705.299035369775 332.59485530546624 724.7459807073956 342.31832797427654 Q 744.1929260450162 352.04180064308684 760.8617363344052 364.54340836012864 Q 777.5305466237943 377.04501607717043 791.4212218649518 392.32475884244377 Q 805.3118971061094 407.60450160771705 816.4244372990354 424.27331189710617 Q 827.5369774919615 440.9421221864952 834.4823151125403 459.00000000000006 Q 841.4276527331191 477.0578778135049 846.9839228295821 506.22829581993574 Q 852.540192926045 535.3987138263666 853.9292604501609 559.0128617363345 Q 855.3183279742766 582.6270096463023 855.3183279742766 609.0192926045017 Q 855.3183279742766 635.411575562701 846.9839228295821 665.9710610932477 Q 838.6495176848875 696.5305466237943 826.1479099678457 728.4790996784566 Q 813.646302250804 760.427652733119 798.3665594855306 790.9871382636657 Q 783.0868167202573 821.5466237942123 771.9742765273313 839.6045016077171 Q 760.8617363344052 857.6623794212219 744.1929260450162 882.6655948553055 Q 727.5241157556271 907.6688102893892 721.9678456591641 914.6141479099679 Q 716.411575562701 921.5594855305467 710.855305466238 925.726688102894 Q 705.299035369775 929.8938906752412 699.742765273312 931.2829581993569 Q 694.1864951768489 932.6720257234728 690.0192926045016 934.0610932475886 Q 685.8520900321544 935.4501607717043 680.2958199356914 935.4501607717043 Q 674.7395498392284 935.4501607717043 666.4051446945339 932.6720257234728 Q 658.0707395498392 929.8938906752412 655.2926045016077 928.5048231511255 Q 652.5144694533763 927.1157556270098 646.9581993569132 924.3376205787782 Q 641.4019292604502 921.5594855305467 635.8456591639872 920.170418006431 Q 630.2893890675242 918.7813504823151 624.7331189710612 917.3922829581994 Q 619.1768488745981 916.0032154340837 613.620578778135 914.6141479099679 Q 608.0643086816721 913.2250803858522 602.5080385852091 913.2250803858522 Q 596.951768488746 913.2250803858522 590.0064308681672 913.2250803858522 Q 583.0610932475885 913.2250803858522 574.726688102894 913.2250803858522 Q 566.3922829581994 913.2250803858522 558.0578778135049 914.6141479099679 Q 549.7234726688104 916.0032154340837 541.3890675241158 920.170418006431 Q 533.0546623794213 924.3376205787782 523.3311897106109 928.5048231511255 Q 513.6077170418007 932.6720257234728 502.4951768488746 936.83922829582 Q 491.3826366559486 941.0064308681673 484.43729903536985 945.1736334405145 Q 477.49196141479104 949.3408360128618 469.1575562700965 950.7299035369776 Q 460.823151125402 952.1189710610934 449.7106109324759 953.5080385852091 Q 438.5980707395499 954.8971061093248 434.4308681672026 954.8971061093248 Q 430.26366559485535 954.8971061093248 424.7073954983923 950.7299035369775 Q 419.1511254019293 946.5627009646303 416.3729903536978 942.395498392283 Q 413.5948553054663 938.2282958199357 410.8167202572348 932.6720257234728 Q 408.03858520900326 927.1157556270098 405.2604501607717 921.5594855305467 Q 402.4823151125402 916.0032154340837 401.09324758842445 911.8360128617364 Q 399.70418006430873 907.6688102893892 399.70418006430873 903.5016077170419 Q 399.70418006430873 899.3344051446946 399.70418006430873 897.9453376205788 L 399.70418006430873 896.5262700964631" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 567.78 590.96)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-567.78, -590.96)" d="M 538.6409324758843 643.7459807073956 Q 538.6109324758843 643.7459807073956 538.6109324758843 642.3569131832799 Q 538.6109324758843 640.967845659164 537.2218649517686 640.967845659164 Q 535.8327974276527 640.967845659164 531.6655948553055 640.967845659164 Q 527.4983922829582 640.967845659164 523.3311897106109 639.5787781350482 Q 519.1639871382637 638.1897106109325 513.6077170418007 635.411575562701 Q 508.05144694533766 632.6334405144695 503.8842443729904 629.855305466238 Q 499.71704180064313 627.0771704180065 494.1607717041801 622.9099678456593 Q 488.60450160771705 618.742765273312 484.4372990353698 614.5755627009647 Q 480.2700964630225 610.4083601286175 478.88102893890675 606.2411575562702 Q 477.49196141479104 602.0739549839229 476.1028938906753 597.9067524115756 Q 474.71382636655954 593.7395498392283 473.32475884244377 590.9614147909967 Q 471.935691318328 588.1832797427653 471.935691318328 584.016077170418 Q 471.935691318328 579.8488745980708 471.935691318328 577.0707395498393 Q 471.935691318328 574.2926045016078 474.7138263665595 570.1254019292605 Q 477.49196141479104 565.9581993569133 483.048231511254 561.790996784566 Q 488.60450160771705 557.6237942122187 494.1607717041801 553.4565916398715 Q 499.71704180064313 549.2893890675242 509.44051446945343 545.122186495177 Q 519.1639871382637 540.9549839228296 530.2765273311898 536.7877813504823 Q 541.3890675241158 532.620578778135 553.8906752411576 528.4533762057878 Q 566.3922829581994 524.2861736334405 576.1157556270098 522.8971061093248 Q 585.83922829582 521.5080385852091 598.3408360128618 521.5080385852091 Q 610.8424437299036 521.5080385852091 619.1768488745981 521.5080385852091 Q 627.5112540192927 521.5080385852091 633.0675241157558 521.5080385852091 Q 638.6237942122187 521.5080385852091 642.790996784566 525.6752411575563 Q 646.9581993569133 529.8424437299036 651.1254019292605 534.0096463022509 Q 655.2926045016078 538.1768488745981 658.0707395498393 543.7331189710612 Q 660.8488745980708 549.2893890675242 660.8488745980708 553.4565916398715 Q 660.8488745980708 557.6237942122187 662.2379421221865 563.1800643086817 Q 663.6270096463023 568.7363344051447 663.6270096463023 572.903536977492 Q 663.6270096463023 577.0707395498392 663.6270096463023 581.2379421221865 Q 663.6270096463023 585.4051446945338 663.6270096463023 588.1832797427653 Q 663.6270096463023 590.9614147909969 663.6270096463023 593.7395498392284 Q 663.6270096463023 596.5176848874598 663.6270096463023 599.2958199356914 Q 663.6270096463023 602.0739549839229 663.6270096463023 603.4630225080386 Q 663.6270096463023 604.8520900321544 663.6270096463023 607.6302250803859 Q 663.6270096463023 610.4083601286175 662.2379421221865 613.1864951768489 Q 660.8488745980708 615.9646302250804 659.4598070739551 618.742765273312 Q 658.0707395498392 621.5209003215434 655.2926045016077 624.299035369775 Q 652.5144694533763 627.0771704180065 651.1254019292605 629.855305466238 Q 649.7363344051447 632.6334405144695 646.9581993569132 635.411575562701 Q 644.1800643086817 638.1897106109325 640.0128617363345 642.3569131832797 Q 635.8456591639872 646.524115755627 633.0675241157558 647.9131832797427 Q 630.2893890675242 649.3022508038586 627.5112540192927 650.6913183279744 Q 624.7331189710611 652.0803858520901 623.3440514469454 653.4694533762058 Q 621.9549839228297 654.8585209003217 620.5659163987139 656.2475884244374 Q 619.1768488745981 657.6366559485531 617.7877813504824 659.0257234726689 Q 616.3987138263666 660.4147909967846 615.0096463022508 660.4147909967846 Q 613.620578778135 660.4147909967846 612.2315112540193 660.4147909967846 Q 610.8424437299036 660.4147909967846 608.0643086816721 660.4147909967846 Q 605.2861736334405 660.4147909967846 602.5080385852091 660.4147909967846 Q 599.7299035369775 660.4147909967846 595.5627009646303 660.4147909967846 Q 591.395498392283 660.4147909967846 585.83922829582 660.4147909967846 Q 580.2829581993569 660.4147909967846 574.726688102894 659.0257234726689 Q 569.170418006431 657.6366559485531 563.6141479099679 654.8585209003215 Q 558.0578778135049 652.0803858520901 553.8906752411576 650.6913183279744 Q 549.7234726688104 649.3022508038586 546.9453376205788 647.9131832797427 Q 544.1672025723473 646.524115755627 541.3890675241157 645.1350482315113 L 538.5809324758843 643.7159807073956" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 398.3 574.29)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-398.3, -574.29)" d="M 385.84350482315114 554.8156591639872 Q 385.81350482315116 554.8456591639872 384.4244372990354 556.2347266881029 Q 383.0353697749197 557.6237942122187 380.2572347266881 559.0128617363345 Q 377.47909967845663 560.4019292604502 374.70096463022514 563.1800643086817 Q 371.9228295819936 565.9581993569133 370.5337620578779 568.7363344051448 Q 369.1446945337621 571.5144694533763 366.3665594855306 574.2926045016077 Q 363.58842443729907 577.0707395498392 362.19935691318335 578.4598070739551 Q 360.8102893890676 579.8488745980708 360.8102893890676 582.6270096463022 Q 360.8102893890676 585.4051446945338 359.4212218649518 585.4051446945338 Q 358.03215434083603 585.4051446945338 358.03215434083603 586.7942122186496 Q 358.03215434083603 588.1832797427653 358.03215434083603 589.572347266881 Q 358.03215434083603 590.9614147909969 358.03215434083603 592.3504823151126 Q 358.03215434083603 593.7395498392283 358.03215434083603 596.5176848874598 Q 358.03215434083603 599.2958199356914 359.4212218649518 600.6848874598072 Q 360.8102893890676 602.0739549839229 362.19935691318335 603.4630225080386 Q 363.58842443729907 604.8520900321544 364.9774919614148 607.6302250803859 Q 366.36655948553056 610.4083601286175 367.75562700964633 611.7974276527332 Q 369.1446945337621 613.1864951768489 370.5337620578779 614.5755627009646 Q 371.9228295819936 615.9646302250804 374.70096463022514 617.3536977491963 Q 377.47909967845663 618.742765273312 378.8681672025724 620.1318327974277 Q 380.2572347266881 621.5209003215434 381.6463022508039 622.9099678456591 Q 383.0353697749197 624.299035369775 385.81350482315116 624.299035369775 Q 388.59163987138265 624.299035369775 389.9807073954984 624.299035369775 Q 391.3697749196142 624.299035369775 392.7588424437299 624.299035369775 Q 394.1479099678457 624.299035369775 395.53697749196147 622.9099678456591 Q 396.9260450160772 621.5209003215434 398.31511254019296 618.742765273312 Q 399.70418006430873 615.9646302250804 402.4823151125402 613.1864951768489 Q 405.2604501607717 610.4083601286175 406.6495176848875 607.6302250803859 Q 408.03858520900326 604.8520900321544 410.8167202572348 600.6848874598071 Q 413.5948553054663 596.5176848874598 414.98392282958207 592.3504823151126 Q 416.3729903536978 588.1832797427653 419.15112540192933 585.4051446945339 Q 421.9292604501608 582.6270096463023 423.3183279742766 579.8488745980708 Q 424.7073954983923 577.0707395498392 426.09646302250803 574.2926045016077 Q 427.4855305466238 571.5144694533763 428.8745980707396 568.7363344051448 Q 430.26366559485535 565.9581993569133 431.6527331189711 564.5691318327974 Q 433.04180064308684 563.1800643086817 434.43086816720256 561.790996784566 Q 435.81993569131834 560.4019292604502 435.81993569131834 559.0128617363345 Q 435.81993569131834 557.6237942122187 437.2090032154341 556.2347266881029 Q 438.5980707395499 554.8456591639872 438.5980707395499 553.4565916398715 Q 438.5980707395499 552.0675241157556 438.5980707395499 550.6784565916399 Q 438.5980707395499 549.2893890675242 439.98713826366566 547.9003215434084 Q 441.3762057877814 546.5112540192927 441.3762057877814 545.122186495177 Q 441.3762057877814 543.7331189710611 441.3762057877814 542.3440514469453 Q 441.3762057877814 540.9549839228296 441.3762057877814 539.5659163987139 Q 441.3762057877814 538.1768488745981 441.3762057877814 536.7877813504824 Q 441.3762057877814 535.3987138263666 441.3762057877814 534.0096463022508 Q 441.3762057877814 532.620578778135 439.98713826366566 531.2315112540193 Q 438.5980707395499 529.8424437299036 437.2090032154341 528.4533762057879 Q 435.81993569131834 527.0643086816721 434.43086816720256 527.0643086816721 Q 433.04180064308684 527.0643086816721 430.2636655948553 527.0643086816721 Q 427.4855305466238 527.0643086816721 424.7073954983923 525.6752411575562 Q 421.9292604501608 524.2861736334405 419.15112540192933 524.2861736334405 Q 416.3729903536978 524.2861736334405 413.59485530546624 524.2861736334405 Q 410.81672025723475 524.2861736334405 408.03858520900326 524.2861736334405 Q 405.2604501607717 524.2861736334405 403.871382636656 524.2861736334405 Q 402.4823151125402 524.2861736334405 401.09324758842445 524.2861736334405 Q 399.70418006430873 524.2861736334405 396.9260450160772 524.2861736334405 Q 394.1479099678457 524.2861736334405 392.7588424437299 524.2861736334405 Q 391.3697749196142 524.2861736334405 389.9807073954984 524.2861736334405 Q 388.59163987138265 524.2861736334405 388.59163987138265 525.6752411575562 Q 388.59163987138265 527.0643086816721 388.59163987138265 528.4533762057879 Q 388.59163987138265 529.8424437299036 387.20257234726694 529.8424437299036 Q 385.81350482315116 529.8424437299036 385.81350482315116 531.2315112540193 Q 385.81350482315116 532.620578778135 384.4244372990354 534.0096463022508 Q 383.0353697749197 535.3987138263666 381.6463022508039 536.7877813504824 Q 380.2572347266881 538.1768488745981 377.4790996784566 540.9549839228296 Q 374.7009646302251 543.7331189710611 373.3118971061093 546.5112540192927 Q 371.9228295819936 549.2893890675242 370.5337620578779 550.6784565916399 Q 369.1446945337621 552.0675241157556 367.75562700964633 554.8456591639872 Q 366.36655948553056 557.6237942122187 364.9774919614148 559.0128617363345 Q 363.58842443729907 560.4019292604502 362.19935691318335 560.4019292604502 Q 360.8102893890676 560.4019292604502 360.8102893890676 561.790996784566 Q 360.8102893890676 563.1800643086817 359.4212218649518 564.5691318327974 Q 358.03215434083603 565.9581993569133 358.03215434083603 567.347266881029 Q 358.03215434083603 568.7363344051447 356.64308681672026 568.7363344051447 L 355.22401929260457 568.7363344051447" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 271.91 797.93)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-271.91, -797.93)" d="M 260.8274276527331 646.524115755627 Q 260.7974276527331 646.524115755627 259.40836012861735 645.1350482315113 Q 258.01929260450163 643.7459807073956 256.63022508038586 643.7459807073956 Q 255.2411575562701 643.7459807073956 252.4630225080386 643.7459807073956 Q 249.6848874598071 643.7459807073956 248.29581993569133 643.7459807073956 Q 246.90675241157558 643.7459807073956 244.12861736334406 643.7459807073956 Q 241.35048231511257 643.7459807073956 238.57234726688105 643.7459807073956 Q 235.79421221864953 643.7459807073956 233.01607717041804 643.7459807073956 Q 230.23794212218652 643.7459807073956 227.459807073955 643.7459807073956 Q 224.6816720257235 643.7459807073956 220.51446945337622 643.7459807073956 Q 216.34726688102896 643.7459807073956 213.56913183279744 643.7459807073956 Q 210.79099678456592 643.7459807073956 206.62379421221866 643.7459807073956 Q 202.4565916398714 643.7459807073956 198.28938906752413 643.7459807073956 Q 194.12218649517686 643.7459807073956 189.9549839228296 645.1350482315113 Q 185.78778135048233 646.524115755627 183.00964630225081 649.3022508038586 Q 180.2315112540193 652.0803858520901 177.45337620578778 653.4694533762058 Q 174.6752411575563 654.8585209003217 169.11897106109325 659.0257234726689 Q 163.56270096463024 663.1929260450162 162.1736334405145 664.5819935691319 Q 160.78456591639872 665.9710610932476 158.00643086816723 667.3601286173634 Q 155.2282958199357 668.7491961414792 149.6720257234727 672.9163987138264 Q 144.11575562700966 677.0836012861737 142.7266881028939 679.8617363344051 Q 141.33762057877814 682.6398713826367 138.55948553054662 685.4180064308682 Q 135.78135048231513 688.1961414790998 134.39228295819936 690.9742765273313 Q 133.00321543408361 693.7524115755627 131.61414790996787 696.5305466237942 Q 130.2250803858521 699.3086816720257 128.83601286173635 702.0868167202573 Q 127.44694533762059 704.8649517684888 126.05787781350483 707.6430868167204 Q 124.66881028938907 710.4212218649518 124.66881028938907 714.5884244372991 Q 124.66881028938907 718.7556270096463 123.27974276527331 721.5337620578779 Q 121.89067524115757 724.3118971061094 121.89067524115757 728.4790996784567 Q 121.89067524115757 732.646302250804 121.89067524115757 738.2025723472669 Q 121.89067524115757 743.7588424437299 121.89067524115757 747.9260450160772 Q 121.89067524115757 752.0932475884244 121.89067524115757 756.2604501607717 Q 121.89067524115757 760.427652733119 121.89067524115757 764.5948553054664 Q 121.89067524115757 768.7620578778136 121.89067524115757 772.9292604501609 Q 121.89067524115757 777.0964630225081 124.66881028938909 779.8745980707397 Q 127.44694533762059 782.6527331189711 128.83601286173635 785.4308681672026 Q 130.2250803858521 788.2090032154341 134.39228295819936 789.5980707395499 Q 138.55948553054662 790.9871382636657 142.7266881028939 793.7652733118971 Q 146.89389067524118 796.5434083601286 151.06109324758845 797.9324758842445 Q 155.2282958199357 799.3215434083602 166.34083601286176 800.7106109324759 Q 177.4533762057878 802.0996784565917 183.00964630225081 802.0996784565917 Q 188.56591639871385 802.0996784565917 195.5112540192926 802.0996784565917 Q 202.4565916398714 802.0996784565917 209.40192926045017 802.0996784565917 Q 216.34726688102896 802.0996784565917 221.90353697749197 802.0996784565917 Q 227.459807073955 802.0996784565917 231.62700964630227 802.0996784565917 Q 235.79421221864953 802.0996784565917 239.9614147909968 800.7106109324759 Q 244.12861736334406 799.3215434083602 246.90675241157558 796.5434083601288 Q 249.6848874598071 793.7652733118972 252.4630225080386 792.3762057877814 Q 255.2411575562701 790.9871382636657 258.01929260450163 788.2090032154342 Q 260.7974276527331 785.4308681672027 263.5755627009646 782.6527331189711 Q 266.35369774919616 779.8745980707396 270.5209003215434 778.4855305466239 Q 274.6881028938907 777.0964630225081 278.85530546623795 774.3183279742766 Q 283.0225080385852 771.540192926045 287.1897106109325 768.7620578778135 Q 291.35691318327974 765.9839228295821 296.9131832797428 764.5948553054664 Q 302.4694533762058 763.2057877813505 306.6366559485531 760.427652733119 Q 310.80385852090035 757.6495176848875 317.74919614147916 754.871382636656 Q 324.6945337620579 752.0932475884244 328.8617363344052 750.7041800643087 Q 333.02893890675244 749.315112540193 338.5852090032155 747.9260450160773 Q 344.14147909967846 746.5369774919615 353.86495176848877 743.7588424437299 Q 363.58842443729907 740.9807073954985 369.14469453376205 739.5916398713828 Q 374.7009646302251 738.2025723472669 378.8681672025724 738.2025723472669 Q 383.0353697749197 738.2025723472669 388.59163987138265 738.2025723472669 Q 394.1479099678457 738.2025723472669 396.9260450160772 738.2025723472669 Q 399.70418006430873 738.2025723472669 402.4823151125402 738.2025723472669 Q 405.2604501607717 738.2025723472669 408.03858520900326 740.9807073954985 Q 410.81672025723475 743.7588424437299 413.59485530546624 746.5369774919615 Q 416.3729903536978 749.315112540193 417.7620578778135 753.4823151125403 Q 419.1511254019293 757.6495176848875 420.54019292604505 761.8167202572348 Q 421.9292604501608 765.9839228295821 421.9292604501608 771.5401929260452 Q 421.9292604501608 777.0964630225081 421.9292604501608 781.2636655948554 Q 421.9292604501608 785.4308681672027 421.9292604501608 789.5980707395499 Q 421.9292604501608 793.7652733118972 420.54019292604505 797.9324758842445 Q 419.1511254019293 802.0996784565917 417.7620578778135 806.266881028939 Q 416.3729903536978 810.4340836012863 412.2057877813505 813.2122186495178 Q 408.03858520900326 815.9903536977492 403.871382636656 820.1575562700965 Q 399.70418006430873 824.3247588424438 395.53697749196147 828.491961414791 Q 391.3697749196142 832.6591639871383 385.81350482315116 835.4372990353698 Q 380.2572347266881 838.2154340836014 376.09003215434086 840.9935691318328 Q 371.9228295819936 843.7717041800644 363.58842443729907 847.9389067524116 Q 355.25401929260454 852.1061093247589 342.75241157556275 856.2733118971062 Q 330.25080385852095 860.4405144694534 321.9163987138264 861.8295819935693 Q 313.58199356913184 863.218649517685 302.4694533762058 865.9967845659164 Q 291.35691318327974 868.774919614148 281.6334405144695 868.774919614148 Q 271.9099678456592 868.774919614148 264.96463022508044 870.1639871382638 Q 258.01929260450163 871.5530546623795 249.6848874598071 871.5530546623795 Q 241.35048231511257 871.5530546623795 234.4051446945338 872.9421221864952 Q 227.459807073955 874.331189710611 221.90353697749197 874.331189710611 Q 216.34726688102896 874.331189710611 212.1800643086817 874.331189710611 Q 208.01286173633443 874.331189710611 203.84565916398716 875.7202572347268 Q 199.6784565916399 877.1093247588425 198.28938906752416 877.1093247588425 Q 196.90032154340838 877.1093247588425 194.12218649517686 878.4983922829583 Q 191.34405144694534 879.887459807074 188.56591639871385 881.2765273311898 Q 185.78778135048233 882.6655948553056 183.00964630225081 884.0546623794213 Q 180.2315112540193 885.443729903537 178.84244372990355 886.8327974276528 Q 177.4533762057878 888.2218649517686 174.6752411575563 891 Q 171.89710610932477 893.7781350482315 170.50803858520902 895.1672025723474 Q 169.11897106109325 896.5562700964631 167.7299035369775 897.9453376205788 Q 166.34083601286176 899.3344051446946 164.95176848874598 900.7234726688104 Q 163.56270096463024 902.1125401929261 163.56270096463024 903.5016077170419 Q 163.56270096463024 904.8906752411576 162.1736334405145 906.2797427652733 Q 160.78456591639872 907.6688102893892 160.78456591639872 909.0578778135049 Q 160.78456591639872 910.4469453376206 160.78456591639872 911.8360128617364 Q 160.78456591639872 913.2250803858522 160.78456591639872 914.6141479099679 Q 160.78456591639872 916.0032154340837 160.78456591639872 917.3922829581994 Q 160.78456591639872 918.7813504823151 160.78456591639872 920.170418006431 Q 160.78456591639872 921.5594855305467 162.1736334405145 922.9485530546624 Q 163.56270096463024 924.3376205787782 164.95176848874598 925.7266881028941 Q 166.34083601286176 927.1157556270098 167.7299035369775 929.8938906752412 Q 169.11897106109325 932.6720257234728 171.89710610932477 934.0610932475886 Q 174.6752411575563 935.4501607717043 178.84244372990355 936.83922829582 Q 183.00964630225081 938.2282958199357 185.78778135048233 941.0064308681673 Q 188.56591639871385 943.7845659163988 192.73311897106112 943.7845659163988 Q 196.90032154340838 943.7845659163988 201.06752411575565 945.1736334405145 Q 205.2347266881029 946.5627009646303 209.40192926045017 947.951768488746 Q 213.56913183279744 949.3408360128618 217.7363344051447 950.7299035369776 Q 221.90353697749197 952.1189710610934 227.459807073955 952.1189710610934 Q 233.01607717041801 952.1189710610934 237.1832797427653 952.1189710610934 Q 241.35048231511257 952.1189710610934 246.90675241157558 952.1189710610934 Q 252.4630225080386 952.1189710610934 256.63022508038586 952.1189710610934 Q 260.7974276527331 952.1189710610934 264.9646302250804 952.1189710610934 Q 269.13183279742765 952.1189710610934 273.29903536977497 949.3408360128618 Q 277.46623794212223 946.5627009646303 281.6334405144695 945.1736334405145 Q 285.80064308681676 943.7845659163988 288.57877813504825 942.3954983922831 Q 291.35691318327974 941.0064308681673 295.524115755627 939.6173633440515 Q 299.6913183279743 938.2282958199357 302.4694533762058 936.83922829582 Q 305.2475884244373 935.4501607717043 309.4147909967846 932.6720257234728 Q 313.58199356913184 929.8938906752412 316.36012861736333 928.5048231511255 Q 319.1382636655949 927.1157556270098 321.9163987138264 925.7266881028941 Q 324.6945337620579 924.3376205787782 326.0836012861737 922.9485530546624 Q 327.4726688102894 921.5594855305467 330.25080385852095 920.170418006431 Q 333.02893890675244 918.7813504823151 334.4180064308682 917.3922829581994 Q 335.80707395498393 916.0032154340837 337.1961414790997 916.0032154340837 Q 338.5852090032155 916.0032154340837 338.5852090032155 914.6141479099679 Q 338.5852090032155 913.2250803858522 339.9742765273312 913.2250803858522 Q 341.363344051447 913.2250803858522 341.363344051447 911.8360128617364 Q 341.363344051447 910.4469453376206 342.75241157556275 910.4469453376206 Q 344.14147909967846 910.4469453376206 345.53054662379424 910.4469453376206 Q 346.91961414791 910.4469453376206 349.6977491961415 910.4469453376206 Q 352.475884244373 910.4469453376206 356.64308681672026 909.0578778135049 Q 360.8102893890676 907.6688102893892 364.97749196141484 906.2797427652733 Q 369.1446945337621 904.8906752411576 373.31189710610937 903.5016077170419 Q 377.47909967845663 902.1125401929261 383.0353697749197 900.7234726688104 Q 388.59163987138265 899.3344051446946 391.3697749196142 897.9453376205788 Q 394.1479099678457 896.5562700964631 395.53697749196147 896.5562700964631 Q 396.9260450160772 896.5562700964631 398.31511254019296 896.5562700964631 Q 399.70418006430873 896.5562700964631 401.09324758842445 896.5562700964631 Q 402.4823151125402 896.5562700964631 403.871382636656 896.5562700964631 Q 405.2604501607717 896.5562700964631 406.6495176848875 896.5562700964631 Q 408.03858520900326 896.5562700964631 409.427652733119 895.1672025723474 Q 410.81672025723475 893.7781350482315 412.2057877813505 893.7781350482315 L 413.62485530546627 893.7781350482315" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 408.04 1007.68)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-408.04, -1007.68)" d="M 458.0750160771704 946.5627009646303 Q 458.04501607717043 946.5627009646303 458.04501607717043 947.951768488746 Q 458.04501607717043 949.3408360128618 456.6559485530547 949.3408360128618 Q 455.26688102893894 949.3408360128618 453.87781350482317 949.3408360128618 Q 452.48874598070745 949.3408360128618 448.3215434083602 950.7299035369776 Q 444.1543408360129 952.1189710610934 439.98713826366566 953.5080385852091 Q 435.81993569131834 954.8971061093248 431.65273311897107 957.6752411575563 Q 427.4855305466238 960.4533762057879 423.31832797427654 961.8424437299036 Q 419.1511254019293 963.2315112540193 416.3729903536978 966.0096463022509 Q 413.5948553054663 968.7877813504824 409.427652733119 971.5659163987139 Q 405.2604501607717 974.3440514469454 402.4823151125402 975.7331189710612 Q 399.70418006430873 977.122186495177 398.31511254019296 979.9003215434084 Q 396.9260450160772 982.6784565916399 394.1479099678457 984.0675241157558 Q 391.3697749196142 985.4565916398715 391.3697749196142 986.8456591639872 Q 391.3697749196142 988.234726688103 389.9807073954984 988.234726688103 Q 388.59163987138265 988.234726688103 387.20257234726694 989.6237942122187 Q 385.81350482315116 991.0128617363345 384.4244372990354 992.4019292604503 Q 383.0353697749197 993.790996784566 381.6463022508039 995.1800643086817 Q 380.2572347266881 996.5691318327976 380.2572347266881 997.9581993569133 Q 380.2572347266881 999.347266881029 378.8681672025724 1000.7363344051448 Q 377.47909967845663 1002.1254019292605 377.47909967845663 1003.5144694533763 Q 377.47909967845663 1004.9035369774921 376.09003215434086 1007.6816720257236 Q 374.7009646302251 1010.4598070739551 373.3118971061093 1011.8488745980708 Q 371.9228295819936 1013.2379421221866 371.9228295819936 1014.6270096463023 Q 371.9228295819936 1016.016077170418 370.5337620578779 1018.7942122186496 Q 369.1446945337621 1021.5723472668811 367.75562700964633 1022.961414790997 Q 366.36655948553056 1024.3504823151127 366.36655948553056 1025.7395498392284 Q 366.36655948553056 1027.1286173633441 364.9774919614148 1028.5176848874598 Q 363.58842443729907 1029.9067524115756 363.58842443729907 1031.2958199356913 Q 363.58842443729907 1032.6848874598072 363.58842443729907 1034.073954983923 Q 363.58842443729907 1035.4630225080386 362.19935691318335 1036.8520900321544 Q 360.8102893890676 1038.24115755627 360.8102893890676 1039.630225080386 Q 360.8102893890676 1041.0192926045017 360.8102893890676 1042.4083601286175 Q 360.8102893890676 1043.7974276527332 360.8102893890676 1045.186495176849 Q 360.8102893890676 1046.5755627009646 360.8102893890676 1047.9646302250803 Q 360.8102893890676 1049.3536977491963 360.8102893890676 1050.742765273312 Q 360.8102893890676 1052.1318327974277 360.8102893890676 1053.5209003215434 Q 360.8102893890676 1054.9099678456591 360.8102893890676 1056.299035369775 Q 360.8102893890676 1057.6881028938908 360.8102893890676 1059.0771704180065 Q 360.8102893890676 1060.4662379421222 360.8102893890676 1061.855305466238 Q 360.8102893890676 1063.2443729903537 360.8102893890676 1064.6334405144694 Q 360.8102893890676 1066.0225080385853 360.8102893890676 1067.411575562701 Q 360.8102893890676 1068.8006430868168 359.4212218649518 1068.8006430868168 L 358.00215434083606 1068.8006430868168" stroke-linecap="round" />
</g>
<g transform="matrix(1 0 0 1 752.53 995.18)"  >
<path style="stroke: rgb(0,0,0); stroke-width: 30; stroke-dasharray: none; stroke-linecap: round; stroke-dashoffset: 0; stroke-linejoin: round; stroke-miterlimit: 10; fill: none; fill-rule: nonzero; opacity: 1;" vector-effect="non-scaling-stroke"  transform=" translate(-752.53, -995.18)" d="M 685.8220900321544 954.8971061093248 Q 685.8520900321544 954.8971061093248 687.2411575562701 954.8971061093248 Q 688.6302250803859 954.8971061093248 691.4083601286175 954.8971061093248 Q 694.1864951768489 954.8971061093248 696.9646302250804 954.8971061093248 Q 699.742765273312 954.8971061093248 708.0771704180065 954.8971061093248 Q 716.411575562701 954.8971061093248 723.3569131832799 954.8971061093248 Q 730.3022508038586 954.8971061093248 734.4694533762058 954.8971061093248 Q 738.6366559485531 954.8971061093248 745.5819935691319 954.8971061093248 Q 752.5273311897107 954.8971061093248 756.694533762058 954.8971061093248 Q 760.8617363344052 954.8971061093248 765.0289389067525 954.8971061093248 Q 769.1961414790998 954.8971061093248 773.363344051447 956.2861736334405 Q 777.5305466237943 957.6752411575563 780.3086816720258 959.0643086816722 Q 783.0868167202573 960.4533762057879 785.8649517684888 961.8424437299036 Q 788.6430868167204 963.2315112540193 791.4212218649518 963.2315112540193 Q 794.1993569131834 963.2315112540193 795.5884244372992 964.620578778135 Q 796.9774919614149 966.0096463022509 798.3665594855306 966.0096463022509 Q 799.7556270096463 966.0096463022509 799.7556270096463 967.3987138263667 Q 799.7556270096463 968.7877813504824 801.144694533762 968.7877813504824 Q 802.5337620578779 968.7877813504824 802.5337620578779 970.1768488745981 Q 802.5337620578779 971.5659163987139 802.5337620578779 972.9549839228296 Q 802.5337620578779 974.3440514469454 803.9228295819937 975.7331189710612 Q 805.3118971061094 977.122186495177 805.3118971061094 979.9003215434084 Q 805.3118971061094 982.6784565916399 805.3118971061094 985.4565916398715 Q 805.3118971061094 988.234726688103 806.7009646302251 989.6237942122187 Q 808.0900321543409 991.0128617363345 808.0900321543409 992.4019292604503 Q 808.0900321543409 993.790996784566 808.0900321543409 996.5691318327974 Q 808.0900321543409 999.347266881029 808.0900321543409 1000.7363344051448 Q 808.0900321543409 1002.1254019292605 808.0900321543409 1003.5144694533763 Q 808.0900321543409 1004.9035369774921 808.0900321543409 1006.2926045016078 Q 808.0900321543409 1007.6816720257235 808.0900321543409 1010.4598070739551 Q 808.0900321543409 1013.2379421221866 809.4790996784566 1014.6270096463023 Q 810.8681672025724 1016.016077170418 810.8681672025724 1017.4051446945339 Q 810.8681672025724 1018.7942122186496 810.8681672025724 1020.1832797427653 Q 810.8681672025724 1021.5723472668811 812.2572347266882 1022.961414790997 Q 813.646302250804 1024.3504823151127 813.646302250804 1025.7395498392284 Q 813.646302250804 1027.1286173633441 813.646302250804 1028.5176848874598 Q 813.646302250804 1029.9067524115756 815.0353697749197 1031.2958199356913 Q 816.4244372990354 1032.6848874598072 816.4244372990354 1034.073954983923 Q 816.4244372990354 1035.4630225080386 817.8135048231511 1035.4630225080386 L 819.2325723472669 1035.4630225080386" stroke-linecap="round" />
</g>
</svg>';
                return "<tcpdf method=\"ImageSVG\" params=\"" .urlencode('@' .$params) . "\" />";
            break;
            default:
                $function = 'write1DBarcode';

                $params = [
                    $value,
                    $typeMap[$codeType] ?? null,
                    '', '',
                    $data['width'] ?? 60,
                    $data['height'] ?? 30,
                    0.4,
                    [
                        'position' => 'S',
                        'border' => false,
                        'padding' => $data['padding'] ?? 0,
                        'fgcolor' => $data['color'] ?? [0, 0, 0],
                        'bgcolor' => $data['bgcolor'] ?? [255, 255, 255],
                        'text' => $data['text'] ?? true,
                        'font' => 'helvetica',
                        'fontsize' => $data['fontsize'] ?? 14,
                        'stretchtext' => 4,
                    ],
                    'N',
                ];
                break;
        }

        $paramsString = urlencode(json_encode($params));

        return "<tcpdf method=\"{$function}\" params=\"{$paramsString}\" />";
    }

    private function replacePlaceholders(string $string, Entity $entity): string
    {
        $newString = $string;

        $attributeList = ['name'];

        foreach ($attributeList as $attribute) {
            $value = (string) ($entity->get($attribute) ?? '');

            $newString = str_replace('{$' . $attribute . '}', $value, $newString);
        }

        return $newString;
    }
}
