define('custom:views/fields/int-extended', ['views/fields/int'], (IntFieldView) => {
    return class extends IntFieldView {

        // Field type name, muss exakt dem Typ in entityDefs entsprechen
        type = 'integerExtended';

        setup () {
            super.setup();
            
            // 💡 WICHTIG: Parameter aus den Optionen laden und in this.params zusammenführen.
            // Dies stellt sicher, dass Parameter, die über die Backend-Feldkonfiguration (Varchar-Feld)
            // an die View übergeben werden, verfügbar sind.
            
            const fieldDef = this.options.def || {};

            // 1. Merge-Definition-Parameter (standardmäßig in fieldDef.params enthalten)
            if (fieldDef.params) {
                Object.assign(this.params, fieldDef.params);
            }

            // 2. Prüfen, ob Parameter direkt in this.options.params übergeben wurden (häufig bei dynamischen Feldern)
            if (this.options.params) {
                Object.assign(this.params, this.options.params);
            }
            
            // --- HILFREICH ZUM DEBUGGEN (später entfernen) ---
            // console.log('DEBUG: Alle geladenen Parameter:', this.params);
            // console.log('DEBUG: Unit-Wert:', this.params.unit);
            // ----------------------------------------------------
        }

        afterRender () {
            super.afterRender();

            // Safety check: model and field name must exist
            if (!this.model || !this.name || typeof this.model.get !== 'function') return;

            // Read value from model and convert to integer
            const value = parseInt(this.model.get(this.name));
            if (isNaN(value)) return;

            // Array of allowed render modes from params
            const renderModes = Array.isArray(this.params.RenderOn) ? this.params.RenderOn : [];

            // Check if current render mode is allowed (e.g. list, detail)
            if (!this.mode || renderModes.length === 0 || !renderModes.includes(this.mode)) return;

            // 💡 KORRIGIERTER ZUGRIFF: Die Unit kommt jetzt über this.params, da sie aus der Konfig stammt
            const rawUnit = this.params.unit; // <-- Greift auf den in setup() geladenen Parameter zu!
            
            // Sicherstellen, dass es ein String ist und nicht leer
            const unit = (typeof rawUnit === 'string' && rawUnit.length > 0) ? rawUnit : '';

            // Format text: integer value + optional unit
            const displayText = value.toString() + (unit ? ' ' + unit : '');
            const $span = $('<span>').text(displayText);

            // Optional: set bold font if isBold is true
            if (this.params.isBold === true) {
                $span.addClass('text-bold');
            }

            // Optional: set italic font if isItalic is true
            if (this.params.isItalic === true) {
                $span.addClass('text-italic');
            }

            // Determine and set label color class based on threshold values
            if (this.params.useColorLabel === true) {
                const labelClass = this.getThresholdClass(
                    value,
                    this.params.labelThresholdLowValue,
                    this.params.labelThresholdLowClass,
                    this.params.labelThresholdHighValue,
                    this.params.labelThresholdHighClass,
                    this.params.labelThresholdDefaultClass
                );
                if (labelClass) $span.addClass(labelClass);
            }

            // Determine and set text color class based on threshold values
            if (this.params.useColorText === true) {
                const textClass = this.getThresholdClass(
                    value,
                    this.params.textThresholdLowValue,
                    this.params.textThresholdLowClass,
                    this.params.textThresholdHighValue,
                    this.params.textThresholdHighClass,
                    this.params.textThresholdDefaultClass
                );
                if (textClass) $span.addClass(textClass);
            }

            // Insert span with formatted value into DOM
            this.$el.empty().append($span);
        }

        /**
         * Returns the appropriate Bootstrap color class based on thresholds
         * @param {number} value - current value
         * @param {number} lowValue - threshold for low range
         * @param {string} lowClass - CSS class for low range
         * @param {number} highValue - threshold for high range
         * @param {string} highClass - CSS class for high range
         * @param {string} defaultClass - CSS class for default range
         * @returns {string|null} CSS class or null
         */
        getThresholdClass(value, lowValue, lowClass, highValue, highClass, defaultClass) {
            if (typeof value !== 'number') return null;

            if (typeof lowValue === 'number' && value < lowValue) {
                return typeof lowClass === 'string' ? lowClass : null;
            }

            if (typeof highValue === 'number' && value > highValue) {
                return typeof highClass === 'string' ? highClass : null;
            }

            return typeof defaultClass === 'string' ? defaultClass : null;
        }

    };
});