<?php
/************************************************************************
 * This file is part of SMS Providers extension for EspoCRM.
 *
 * Copyright (C) 2014-2022 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
 * Website: https://www.espocrm.com
 *
 * EspoCRM is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * EspoCRM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with EspoCRM. If not, see http://www.gnu.org/licenses/.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
 ************************************************************************/

namespace Espo\Modules\SmsProviders\Rcsrds;

use Espo\Core\Sms\Sender;
use Espo\Core\Sms\Sms;

use Espo\Core\Utils\Config;
use Espo\Core\Utils\Log;
use Espo\Core\Utils\Json;
use Espo\Core\Exceptions\Error;

use Espo\ORM\EntityManager;

use Espo\Entities\Integration;

use Throwable;

class RcsrdsSender implements Sender
{
    private const BASE_URL = 'https://smsapi.rcs-rds.ro/jsonrpc/public/sms/v2';

    private const TIMEOUT = 10;

    public function __construct(
        private Config $config, 
        private EntityManager $em, 
        private Log $log)
    {}

    public function send(Sms $sms): void
    {
        $toNumberList = $sms->getToNumberList();

        if (!count($toNumberList)) {
            throw new Error("No recipient phone number.");
        }

        foreach ($toNumberList as $number) {
            $this->sendToNumber($sms, $number);
        }
    }

    private function sendToNumber(Sms $sms, string $toNumber): void
    {
        $integration = $this->getIntegrationEntity();

        $apiKey = $integration->get('rcsrdsApiKey');
        $basicLogin = $integration->get('rcsrdsBasicLogin');
        $basicPassword = $integration->get('rcsrdsBasicPassword');
        $baseUrl = rtrim( $integration->get('rcsrdsApiBaseUrl') ?? self::BASE_URL);
        $originator = $integration->get('rcsrdsOriginator');
        $method = $integration->get('rcsrdsMethod');
        $logEnabled = $integration->get('rcsrdsLogEnabled');

        $timeout = $this->config->get('rcsrdsSmsSendTimeout') ?? self::TIMEOUT;

        if (!$basicLogin) {
            throw new Error("No Rcsrds Basic Login.");
        }
        if (!$basicPassword) {
            throw new Error("No Rcsrds Basic Password.");
        }

        if (!$toNumber) {
            throw new Error("No recipient phone number.");
        }

        $data = [
            'jsonrpc' => '2.0',
            'id' => 1,
            'method' => $method,
            'params' => [
                    'smsRequest' => [
                            'originator' => $originator,
                            'recipient' => self::formatNumber($toNumber),
                            'text' => $sms->getBody(),
                        ]
                ]
        ];

        $headers = [
            //'X-Api-Key: ' . $apiKey,
            'Content-Type: application/json',
            'Accept: application/json',
            'Authorization: Basic '. base64_encode("{$basicLogin}:{$basicPassword}") 
        ];

        $ch = curl_init();

        curl_setopt($ch, \CURLOPT_URL, $baseUrl);
        curl_setopt($ch, \CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, \CURLOPT_HEADER, true);
        curl_setopt($ch, \CURLOPT_TIMEOUT, $timeout);
        curl_setopt($ch, \CURLOPT_CONNECTTIMEOUT, $timeout);
        curl_setopt($ch, \CURLOPT_CUSTOMREQUEST, 'POST');
        curl_setopt($ch, \CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, \CURLOPT_POSTFIELDS, $this->buildQuery($data));


        $response = curl_exec($ch);
        $code = curl_getinfo($ch, \CURLINFO_HTTP_CODE);
        $error = curl_errno($ch);

        if ($code) {
            $headerSize = curl_getinfo($ch, \CURLINFO_HEADER_SIZE);
            $body = mb_substr($response, $headerSize);
            $obj = Json::decode($body);

            $errorCode = isset($obj->result->errorCode) ? (int)$obj->result->errorCode : null;

            if (0 !== $errorCode) {
                $this->processError($code, $errorCode);
            }else{
                if ($logEnabled){
                    $this->log->error(Json::encode($body));
                }
            }
        }

        if ($error) {
            if (in_array($error, [\CURLE_OPERATION_TIMEDOUT, \CURLE_OPERATION_TIMEOUTED])) {
                throw new Error("Rcsrds SMS sending timeout.");
            }
        }
    }

    private function buildQuery(array $data): string
    {
        return json_encode($data);
    }

    private static function formatNumber(string $number): string
    {
        return '+' . preg_replace('/[^0-9]/', '', $number);
    }

    private function processError(int $code, int $errorCode): void
    {

        switch($errorCode) {
            case 0:
                $message = 'SUCCESS';
                break;
            case 1:
                $message = 'INVALID_SYSTEMID';
                break;
            case 2:
                $message = 'INVALID_ALIAS';
                break;
            case 3:
                $message = 'INVALID_DCS';
                break;
            case 4:
                $message = 'INVALID_TEXT';
                break;
            case 5:
                $message = 'INVALID_RECIPIENT';
                break;
            case 6:
                $message = 'INVALID_RECIPIENT_IS_BLACKLISTED';
                break;
            case 7:
                $message = 'INVALID_RECIPIENT_NOT_ALLOWED';
                break;
            case 8:
                $message = 'MAX_QUOTA_REACHED';
                break;
            case 9:
                $message = 'SMS_NOT_FOUND';
                break;
            case 10:
                $message = 'SMS_NOT_CANCELABLE';
                break;
            case 11:
                $message = 'TOO_MANY_PARAMETERS';
                break;
            case 12:
                $message = 'SYSTEM_ERROR';
                break;
            case 13:
                $message = 'USER_NOT_FOUND';
                break;
            case 14:
                $message = 'INVALID_PRIORITY';
                break;
            case -32001:
                $message = 'INVALID_SYSTEMID';
                break;
            case -32002:
                $message = 'INVALID_ALIAS';
                break;
            case -32003:
                $message = 'INVALID_DCS';
                break;
            case -32004:
                $message = 'INVALID_TEXT';
                break;
            case -32005:
                $message = 'INVALID_RECIPIENT';
                break;
            case -32006:
                $message = 'INVALID_RECIPIENT_IS_BLACKLISTED';
                break;
            case -32007:
                $message = 'INVALID_RECIPIENT_NOT_ALLOWED';
                break;
            case -32008:
                $message = 'MAX_QUOTA_REACHED';
                break;
            case -32009:
                $message = 'SMS_NOT_FOUND';
                break;
            case -32010:
                $message = 'SMS_NOT_CANCELABLE';
                break;
            case -32011:
                $message = 'TOO_MANY_PARAMETERS';
                break;
            case -32012:
                $message = 'SYSTEM_ERROR';
                break;
            case -32013:
                $message = 'USER_NOT_FOUND';
                break;
            case -32014:
                $message = 'INVALID_PRIORITY';
                break;
            case -32015:
                $message = 'AUTH_FAILURE';
                break;
            case -32097:
                $message = 'AUTH_FAILURE';
                break;
            case -32098:
                $message = 'ACCESS_DENIED';
                break;
            case -32099:
                $message = 'TOO_MANY_REQUESTS';
                break;
            case -32603:
                $message = 'TOO_MANY_REQUESTS';
                break;
            default:
                $message = 'UNKWNOW_ERROR';

        }

        $this->log->error("Rcsrds SMS sending error. Message: {$code} : " . $message);

    }

    private function getIntegrationEntity(): Integration
    {
        $entity = $this->em->getEntity(Integration::ENTITY_TYPE, 'Rcsrds');

        if (!$entity || !$entity->get('enabled')) {
            throw new Error("Rcsrds integration is not enabled");
        }
        return $entity;
    }
}
