<?php

require_once('cloudmail_keys.php');

/**
 * Cloud Mail 2024 PHP Edition - OAuth Component
 *
 * Copyright (c) 2025 /n software inc.
 *
 * For more information, please visit http://www.nsoftware.com/.
 *
 */

global $lib_cloudmail_oauth;
global $objMap;

$lib_cloudmail_oauth = FFI::cdef("
  typedef int (*CloudMail_OAuth_callback) (void *pobj, int event_id, int cparam, void *param[], int cbparam[]);
  void* CloudMail_OAuth_Create(CloudMail_OAuth_callback lpSink, void *lpContext, char *lpOemKey, int opts);
  int   CloudMail_OAuth_Destroy(void *lpObj);
  int   CloudMail_OAuth_CheckIndex(void *lpObj, int propid, int arridx);
  void* CloudMail_OAuth_Get(void *lpObj, int propid, int arridx, int *lpcbVal, int64_t *lpllVal);
  int   CloudMail_OAuth_Set(void *lpObj, int propid, int arridx, const void *val, int cbVal);
  int   CloudMail_OAuth_Do(void *lpObj, int methid, int cparam, void *param[], int cbparam[], int64_t *lpllVal);
  char* CloudMail_OAuth_GetLastError(void *lpObj);
  int   CloudMail_OAuth_GetLastErrorCode(void *lpObj);
  void* CloudMail_EvtStr(void *lpEvtStr, int id, void *val, int opt);",
  "libcloudmail.so.24.0");

$objMap = [];

function cloudmail_oauth_callback($lpObj, $event_id, $cparam, $param, $cbparam) {
  global $lib_cloudmail_oauth;
  global $objMap;
  if (FFI::isnull($lpObj)) return -1;

  $objId = $lpObj - $lib_cloudmail_oauth->cast("void*", 0);
  $obj = $objMap[$objId];
  if (!isset($obj)) throw new Exception("invalid php object id");

  switch ($event_id) {
    case 1: { /* CONNECTED */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["statuscode"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["statuscode"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["description"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
      }

      $retval = $obj->fireConnected($param_array);

 
      break;
    }

    case 2: { /* CONNECTIONSTATUS */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["connectionevent"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        $param_array["statuscode"] = 0;
        if (!FFI::isnull($param[1])) {
          $param_array["statuscode"] = $param[1] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[2])) {
          $param_array["description"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireConnectionStatus($param_array);

 
      break;
    }

    case 3: { /* DISCONNECTED */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["statuscode"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["statuscode"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["description"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
      }

      $retval = $obj->fireDisconnected($param_array);

 
      break;
    }

    case 4: { /* ENDTRANSFER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["direction"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["direction"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
      }

      $retval = $obj->fireEndTransfer($param_array);

 
      break;
    }

    case 5: { /* ERROR */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["errorcode"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["errorcode"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["description"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
      }

      $retval = $obj->fireError($param_array);

 
      break;
    }

    case 6: { /* HEADER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["field"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["value"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
      }

      $retval = $obj->fireHeader($param_array);

 
      break;
    }

    case 7: { /* LAUNCHBROWSER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $str = $lib_cloudmail_oauth->CloudMail_EvtStr($param[0], 0, null, 1);
          if (isset($str)) $param_array["url"] = FFI::string($lib_cloudmail_oauth->cast("char*", $str));
        }
        if (!FFI::isnull($param[1])) {
          $str = $lib_cloudmail_oauth->CloudMail_EvtStr($param[1], 0, null, 1);
          if (isset($str)) $param_array["command"] = FFI::string($lib_cloudmail_oauth->cast("char*", $str));
        }
      }

      $retval = $obj->fireLaunchBrowser($param_array);

      if (isset($retval["url"])) {
        $lib_cloudmail_oauth->CloudMail_EvtStr($param[0], 2, (string)$retval["url"], 1);
      }
      if (isset($retval["command"])) {
        $lib_cloudmail_oauth->CloudMail_EvtStr($param[1], 2, (string)$retval["command"], 1);
      }
 
      break;
    }

    case 8: { /* LOG */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["loglevel"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["loglevel"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["message"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["logtype"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireLog($param_array);

 
      break;
    }

    case 9: { /* REDIRECT */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["location"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        $param_array["accept"] = !FFI::isnull($param[1]);
      }

      $retval = $obj->fireRedirect($param_array);

      if (isset($retval["accept"])) {
        $param[1] = $lib_cloudmail_oauth->cast("void*", (int)$retval["accept"]);
      }
 
      break;
    }

    case 10: { /* RETURNURL */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["urlpath"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["querystring"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $str = $lib_cloudmail_oauth->CloudMail_EvtStr($param[2], 0, null, 1);
          if (isset($str)) $param_array["responseheaders"] = FFI::string($lib_cloudmail_oauth->cast("char*", $str));
        }
        if (!FFI::isnull($param[3])) {
          $str = $lib_cloudmail_oauth->CloudMail_EvtStr($param[3], 0, null, 1);
          if (isset($str)) $param_array["responsebody"] = FFI::string($lib_cloudmail_oauth->cast("char*", $str));
        }
      }

      $retval = $obj->fireReturnURL($param_array);

      if (isset($retval["responseheaders"])) {
        $lib_cloudmail_oauth->CloudMail_EvtStr($param[2], 2, (string)$retval["responseheaders"], 1);
      }
      if (isset($retval["responsebody"])) {
        $lib_cloudmail_oauth->CloudMail_EvtStr($param[3], 2, (string)$retval["responsebody"], 1);
      }
 
      break;
    }

    case 11: { /* SETCOOKIE */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["name"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["value"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["expires"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["domain"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[3]));
        }
        if (!FFI::isnull($param[4])) {
          $param_array["path"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[4]));
        }
        $param_array["secure"] = !FFI::isnull($param[5]);
      }

      $retval = $obj->fireSetCookie($param_array);

 
      break;
    }

    case 12: { /* SSLSERVERAUTHENTICATION */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["certencoded"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]), (int)$cbparam[0]);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["certsubject"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["certissuer"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["status"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[3]));
        }
        $param_array["accept"] = !FFI::isnull($param[4]);
      }

      $retval = $obj->fireSSLServerAuthentication($param_array);

      if (isset($retval["accept"])) {
        $param[4] = $lib_cloudmail_oauth->cast("void*", (int)$retval["accept"]);
      }
 
      break;
    }

    case 13: { /* SSLSTATUS */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["message"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
      }

      $retval = $obj->fireSSLStatus($param_array);

 
      break;
    }

    case 14: { /* STARTTRANSFER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["direction"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["direction"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
      }

      $retval = $obj->fireStartTransfer($param_array);

 
      break;
    }

    case 15: { /* STATUS */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["httpversion"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));
        }
        $param_array["statuscode"] = 0;
        if (!FFI::isnull($param[1])) {
          $param_array["statuscode"] = $param[1] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[2])) {
          $param_array["description"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireStatus($param_array);

 
      break;
    }

    case 16: { /* TRANSFER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["direction"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["direction"] = $param[0] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $retLong64Val = $lib_cloudmail_oauth->cast("int64_t*", $param[1]);
          $param_array["bytestransferred"] = (string)$retLong64Val[0];
        }
        $param_array["percentdone"] = 0;
        if (!FFI::isnull($param[2])) {
          $param_array["percentdone"] = $param[2] - $lib_cloudmail_oauth->cast("void*", 0);
        }
        if (!FFI::isnull($param[3])) {
          $param_array["text"] = FFI::string($lib_cloudmail_oauth->cast("char*", $param[3]), (int)$cbparam[3]);
        }
      }

      $retval = $obj->fireTransfer($param_array);

 
      break;
    }

  }

  return 0;
}

class CloudMail_OAuth {
  
  var $handle;

  public function __construct() {
    $this->cloudmail_oauth_open(CLOUDMAIL_OEMKEY_75);
  }

  public function __destruct() {
    $this->cloudmail_oauth_close();
  }


 /**
  * Returns the last error code.
  *
  * @access   public
  */
  public function lastError() {
    return $this->cloudmail_oauth_get_last_error();
  }

  
 /**
  * Returns the last error message.
  *
  * @access   public
  */
  public function lastErrorCode() {
    return $this->cloudmail_oauth_get_last_error_code();
  }


 /**
  * Adds a cookie and the corresponding value to the outgoing request headers.
  *
  * @access   public
  * @param    string    cookiename

  * @param    string    cookievalue

  */
  public function doAddCookie($cookiename, $cookievalue) {
    $ret = $this->cloudmail_oauth_do_addcookie($cookiename, $cookievalue);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Adds a name-value pair to the query string parameters of the outgoing request.
  *
  * @access   public
  * @param    string    paramname

  * @param    string    paramvalue

  */
  public function doAddParam($paramname, $paramvalue) {
    $ret = $this->cloudmail_oauth_do_addparam($paramname, $paramvalue);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Sets or retrieves a configuration setting.
  *
  * @access   public
  * @param    string    configurationstring

  */
  public function doConfig($configurationstring) {
    $ret = $this->cloudmail_oauth_do_config($configurationstring);
    $err = $this->cloudmail_oauth_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * This method processes events from the internal message queue.
  *
  * @access   public
  */
  public function doEvents() {
    $ret = $this->cloudmail_oauth_do_doevents();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Gets the authorization string required to access the protected resource.
  *
  * @access   public
  */
  public function doGetAuthorization() {
    $ret = $this->cloudmail_oauth_do_getauthorization();
    $err = $this->cloudmail_oauth_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Builds and returns the URL to which the user should be redirected for authorization.
  *
  * @access   public
  */
  public function doGetAuthorizationURL() {
    $ret = $this->cloudmail_oauth_do_getauthorizationurl();
    $err = $this->cloudmail_oauth_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Gets a specific parameter from a query string.
  *
  * @access   public
  * @param    string    paramname

  */
  public function doGetParam($paramname) {
    $ret = $this->cloudmail_oauth_do_getparam($paramname);
    $err = $this->cloudmail_oauth_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * This method interrupts the current method.
  *
  * @access   public
  */
  public function doInterrupt() {
    $ret = $this->cloudmail_oauth_do_interrupt();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * This method will reset the class.
  *
  * @access   public
  */
  public function doReset() {
    $ret = $this->cloudmail_oauth_do_reset();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Starts the embedded web server.
  *
  * @access   public
  */
  public function doStartWebServer() {
    $ret = $this->cloudmail_oauth_do_startwebserver();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


 /**
  * Stops the embedded web server.
  *
  * @access   public
  */
  public function doStopWebServer() {
    $ret = $this->cloudmail_oauth_do_stopwebserver();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }   


   

  public function VERSION() {
    return $this->cloudmail_oauth_get(0);
  }

 /**
  * The access token returned by the authorization server.
  *
  * @access   public
  */
  public function getAccessToken() {
    $ret = $this->cloudmail_oauth_get(1 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The access token returned by the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setAccessToken($value) {
    $ret = $this->cloudmail_oauth_set(1, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The lifetime of the access token.
  *
  * @access   public
  */
  public function getAccessTokenExp() {
    $ret = $this->cloudmail_oauth_get(2 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The authorization code that is exchanged for an access token.
  *
  * @access   public
  */
  public function getAuthorizationCode() {
    $ret = $this->cloudmail_oauth_get(3 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The authorization code that is exchanged for an access token.
  *
  * @access   public
  * @param    string   value
  */
  public function setAuthorizationCode($value) {
    $ret = $this->cloudmail_oauth_set(3, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The scope request or response parameter used during authorization.
  *
  * @access   public
  */
  public function getAuthorizationScope() {
    $ret = $this->cloudmail_oauth_get(4 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The scope request or response parameter used during authorization.
  *
  * @access   public
  * @param    string   value
  */
  public function setAuthorizationScope($value) {
    $ret = $this->cloudmail_oauth_set(4, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The Id of the client assigned when registering the application.
  *
  * @access   public
  */
  public function getClientId() {
    $ret = $this->cloudmail_oauth_get(5 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The Id of the client assigned when registering the application.
  *
  * @access   public
  * @param    string   value
  */
  public function setClientId($value) {
    $ret = $this->cloudmail_oauth_set(5, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of client that is requesting authorization.
  *
  * @access   public
  */
  public function getClientProfile() {
    $ret = $this->cloudmail_oauth_get(6 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of client that is requesting authorization.
  *
  * @access   public
  * @param    int   value
  */
  public function setClientProfile($value) {
    $ret = $this->cloudmail_oauth_set(6, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The secret value for the client assigned when registering the application.
  *
  * @access   public
  */
  public function getClientSecret() {
    $ret = $this->cloudmail_oauth_get(7 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The secret value for the client assigned when registering the application.
  *
  * @access   public
  * @param    string   value
  */
  public function setClientSecret($value) {
    $ret = $this->cloudmail_oauth_set(7, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether the class is connected.
  *
  * @access   public
  */
  public function getConnected() {
    $ret = $this->cloudmail_oauth_get(8 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the Cookie arrays.
  *
  * @access   public
  */
  public function getCookieCount() {
    $ret = $this->cloudmail_oauth_get(9 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Cookie arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setCookieCount($value) {
    $ret = $this->cloudmail_oauth_set(9, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The domain of a received cookie.
  *
  * @access   public
  */
  public function getCookieDomain($cookieindex) {
    $ret = $this->cloudmail_oauth_get(10 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * An expiration time for the cookie (if provided by the server).
  *
  * @access   public
  */
  public function getCookieExpiration($cookieindex) {
    $ret = $this->cloudmail_oauth_get(11 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the cookie.
  *
  * @access   public
  */
  public function getCookieName($cookieindex) {
    $ret = $this->cloudmail_oauth_get(12 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the cookie.
  *
  * @access   public
  * @param    string   value
  */
  public function setCookieName($cookieindex, $value) {
    $ret = $this->cloudmail_oauth_set(12, $value , $cookieindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A path name to limit the cookie to (if provided by the server).
  *
  * @access   public
  */
  public function getCookiePath($cookieindex) {
    $ret = $this->cloudmail_oauth_get(13 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The security flag of the received cookie.
  *
  * @access   public
  */
  public function getCookieSecure($cookieindex) {
    $ret = $this->cloudmail_oauth_get(14 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The value of the cookie.
  *
  * @access   public
  */
  public function getCookieValue($cookieindex) {
    $ret = $this->cloudmail_oauth_get(15 , $cookieindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The value of the cookie.
  *
  * @access   public
  * @param    string   value
  */
  public function setCookieValue($cookieindex, $value) {
    $ret = $this->cloudmail_oauth_set(15, $value , $cookieindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether to automatically detect and use firewall system settings, if available.
  *
  * @access   public
  */
  public function getFirewallAutoDetect() {
    $ret = $this->cloudmail_oauth_get(16 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether to automatically detect and use firewall system settings, if available.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setFirewallAutoDetect($value) {
    $ret = $this->cloudmail_oauth_set(16, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of firewall to connect through.
  *
  * @access   public
  */
  public function getFirewallType() {
    $ret = $this->cloudmail_oauth_get(17 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of firewall to connect through.
  *
  * @access   public
  * @param    int   value
  */
  public function setFirewallType($value) {
    $ret = $this->cloudmail_oauth_set(17, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name or IP address of the firewall (optional).
  *
  * @access   public
  */
  public function getFirewallHost() {
    $ret = $this->cloudmail_oauth_get(18 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name or IP address of the firewall (optional).
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallHost($value) {
    $ret = $this->cloudmail_oauth_set(18, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A password if authentication is to be used when connecting through the firewall.
  *
  * @access   public
  */
  public function getFirewallPassword() {
    $ret = $this->cloudmail_oauth_get(19 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * A password if authentication is to be used when connecting through the firewall.
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallPassword($value) {
    $ret = $this->cloudmail_oauth_set(19, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The Transmission Control Protocol (TCP) port for the firewall Host .
  *
  * @access   public
  */
  public function getFirewallPort() {
    $ret = $this->cloudmail_oauth_get(20 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The Transmission Control Protocol (TCP) port for the firewall Host .
  *
  * @access   public
  * @param    int   value
  */
  public function setFirewallPort($value) {
    $ret = $this->cloudmail_oauth_set(20, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A username if authentication is to be used when connecting through a firewall.
  *
  * @access   public
  */
  public function getFirewallUser() {
    $ret = $this->cloudmail_oauth_get(21 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * A username if authentication is to be used when connecting through a firewall.
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallUser($value) {
    $ret = $this->cloudmail_oauth_set(21, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Determines what happens when the server issues a redirect.
  *
  * @access   public
  */
  public function getFollowRedirects() {
    $ret = $this->cloudmail_oauth_get(22 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Determines what happens when the server issues a redirect.
  *
  * @access   public
  * @param    int   value
  */
  public function setFollowRedirects($value) {
    $ret = $this->cloudmail_oauth_set(22, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The OAuth grant type used to acquire an OAuth access token.
  *
  * @access   public
  */
  public function getGrantType() {
    $ret = $this->cloudmail_oauth_get(23 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The OAuth grant type used to acquire an OAuth access token.
  *
  * @access   public
  * @param    int   value
  */
  public function setGrantType($value) {
    $ret = $this->cloudmail_oauth_set(23, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The current status of the class.
  *
  * @access   public
  */
  public function getIdle() {
    $ret = $this->cloudmail_oauth_get(24 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the local host or user-assigned IP interface through which connections are initiated or accepted.
  *
  * @access   public
  */
  public function getLocalHost() {
    $ret = $this->cloudmail_oauth_get(25 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the local host or user-assigned IP interface through which connections are initiated or accepted.
  *
  * @access   public
  * @param    string   value
  */
  public function setLocalHost($value) {
    $ret = $this->cloudmail_oauth_set(25, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Other headers as determined by the user (optional).
  *
  * @access   public
  */
  public function getOtherHeaders() {
    $ret = $this->cloudmail_oauth_get(26 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Other headers as determined by the user (optional).
  *
  * @access   public
  * @param    string   value
  */
  public function setOtherHeaders($value) {
    $ret = $this->cloudmail_oauth_set(26, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the Param arrays.
  *
  * @access   public
  */
  public function getParamCount() {
    $ret = $this->cloudmail_oauth_get(27 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Param arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setParamCount($value) {
    $ret = $this->cloudmail_oauth_set(27, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the parameter to be used in the request or returned in the response.
  *
  * @access   public
  */
  public function getParamName($paramindex) {
    $ret = $this->cloudmail_oauth_get(28 , $paramindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the parameter to be used in the request or returned in the response.
  *
  * @access   public
  * @param    string   value
  */
  public function setParamName($paramindex, $value) {
    $ret = $this->cloudmail_oauth_set(28, $value , $paramindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The value of the parameter to be used in the request or returned in the response.
  *
  * @access   public
  */
  public function getParamValue($paramindex) {
    $ret = $this->cloudmail_oauth_get(29 , $paramindex);
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The value of the parameter to be used in the request or returned in the response.
  *
  * @access   public
  * @param    string   value
  */
  public function setParamValue($paramindex, $value) {
    $ret = $this->cloudmail_oauth_set(29, $value , $paramindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of authorization to perform when connecting to the proxy.
  *
  * @access   public
  */
  public function getProxyAuthScheme() {
    $ret = $this->cloudmail_oauth_get(30 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of authorization to perform when connecting to the proxy.
  *
  * @access   public
  * @param    int   value
  */
  public function setProxyAuthScheme($value) {
    $ret = $this->cloudmail_oauth_set(30, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether to automatically detect and use proxy system settings, if available.
  *
  * @access   public
  */
  public function getProxyAutoDetect() {
    $ret = $this->cloudmail_oauth_get(31 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether to automatically detect and use proxy system settings, if available.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setProxyAutoDetect($value) {
    $ret = $this->cloudmail_oauth_set(31, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A password if authentication is to be used for the proxy.
  *
  * @access   public
  */
  public function getProxyPassword() {
    $ret = $this->cloudmail_oauth_get(32 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * A password if authentication is to be used for the proxy.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyPassword($value) {
    $ret = $this->cloudmail_oauth_set(32, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The Transmission Control Protocol (TCP) port for the proxy Server (default 80).
  *
  * @access   public
  */
  public function getProxyPort() {
    $ret = $this->cloudmail_oauth_get(33 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The Transmission Control Protocol (TCP) port for the proxy Server (default 80).
  *
  * @access   public
  * @param    int   value
  */
  public function setProxyPort($value) {
    $ret = $this->cloudmail_oauth_set(33, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * If a proxy Server is given, then the HTTP request is sent to the proxy instead of the server otherwise specified.
  *
  * @access   public
  */
  public function getProxyServer() {
    $ret = $this->cloudmail_oauth_get(34 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * If a proxy Server is given, then the HTTP request is sent to the proxy instead of the server otherwise specified.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyServer($value) {
    $ret = $this->cloudmail_oauth_set(34, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * When to use a Secure Sockets Layer (SSL) for the connection to the proxy.
  *
  * @access   public
  */
  public function getProxySSL() {
    $ret = $this->cloudmail_oauth_get(35 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * When to use a Secure Sockets Layer (SSL) for the connection to the proxy.
  *
  * @access   public
  * @param    int   value
  */
  public function setProxySSL($value) {
    $ret = $this->cloudmail_oauth_set(35, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A username if authentication is to be used for the proxy.
  *
  * @access   public
  */
  public function getProxyUser() {
    $ret = $this->cloudmail_oauth_get(36 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * A username if authentication is to be used for the proxy.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyUser($value) {
    $ret = $this->cloudmail_oauth_set(36, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The refresh token received from or sent to the authorization server.
  *
  * @access   public
  */
  public function getRefreshToken() {
    $ret = $this->cloudmail_oauth_get(37 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The refresh token received from or sent to the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setRefreshToken($value) {
    $ret = $this->cloudmail_oauth_set(37, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL where the user (browser) returns after authenticating.
  *
  * @access   public
  */
  public function getReturnURL() {
    $ret = $this->cloudmail_oauth_get(38 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL where the user (browser) returns after authenticating.
  *
  * @access   public
  * @param    string   value
  */
  public function setReturnURL($value) {
    $ret = $this->cloudmail_oauth_set(38, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL of the authorization server.
  *
  * @access   public
  */
  public function getServerAuthURL() {
    $ret = $this->cloudmail_oauth_get(39 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL of the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setServerAuthURL($value) {
    $ret = $this->cloudmail_oauth_set(39, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL used to obtain the access token.
  *
  * @access   public
  */
  public function getServerTokenURL() {
    $ret = $this->cloudmail_oauth_get(40 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL used to obtain the access token.
  *
  * @access   public
  * @param    string   value
  */
  public function setServerTokenURL($value) {
    $ret = $this->cloudmail_oauth_set(40, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertEffectiveDate() {
    $ret = $this->cloudmail_oauth_get(41 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertExpirationDate() {
    $ret = $this->cloudmail_oauth_get(42 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertExtendedKeyUsage() {
    $ret = $this->cloudmail_oauth_get(43 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprint() {
    $ret = $this->cloudmail_oauth_get(44 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprintSHA1() {
    $ret = $this->cloudmail_oauth_get(45 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprintSHA256() {
    $ret = $this->cloudmail_oauth_get(46 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertIssuer() {
    $ret = $this->cloudmail_oauth_get(47 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKey() {
    $ret = $this->cloudmail_oauth_get(48 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_oauth_get(49 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKeyContainer() {
    $ret = $this->cloudmail_oauth_get(50 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKey() {
    $ret = $this->cloudmail_oauth_get(51 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_oauth_get(52 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKeyLength() {
    $ret = $this->cloudmail_oauth_get(53 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSerialNumber() {
    $ret = $this->cloudmail_oauth_get(54 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSignatureAlgorithm() {
    $ret = $this->cloudmail_oauth_get(55 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStore() {
    $ret = $this->cloudmail_oauth_get(56 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertStore($value) {
    $ret = $this->cloudmail_oauth_set(56, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStorePassword() {
    $ret = $this->cloudmail_oauth_get(57 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertStorePassword($value) {
    $ret = $this->cloudmail_oauth_set(57, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStoreType() {
    $ret = $this->cloudmail_oauth_get(58 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLAcceptServerCertStoreType($value) {
    $ret = $this->cloudmail_oauth_set(58, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSubjectAltNames() {
    $ret = $this->cloudmail_oauth_get(59 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintMD5() {
    $ret = $this->cloudmail_oauth_get(60 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintSHA1() {
    $ret = $this->cloudmail_oauth_get(61 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintSHA256() {
    $ret = $this->cloudmail_oauth_get(62 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLAcceptServerCertUsage() {
    $ret = $this->cloudmail_oauth_get(63 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertUsageFlags() {
    $ret = $this->cloudmail_oauth_get(64 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertVersion() {
    $ret = $this->cloudmail_oauth_get(65 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSubject() {
    $ret = $this->cloudmail_oauth_get(66 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertSubject($value) {
    $ret = $this->cloudmail_oauth_set(66, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertEncoded() {
    $ret = $this->cloudmail_oauth_get(67 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertEncoded($value) {
    $ret = $this->cloudmail_oauth_set(67, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLCertEffectiveDate() {
    $ret = $this->cloudmail_oauth_get(68 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLCertExpirationDate() {
    $ret = $this->cloudmail_oauth_get(69 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLCertExtendedKeyUsage() {
    $ret = $this->cloudmail_oauth_get(70 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprint() {
    $ret = $this->cloudmail_oauth_get(71 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprintSHA1() {
    $ret = $this->cloudmail_oauth_get(72 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprintSHA256() {
    $ret = $this->cloudmail_oauth_get(73 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLCertIssuer() {
    $ret = $this->cloudmail_oauth_get(74 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLCertPrivateKey() {
    $ret = $this->cloudmail_oauth_get(75 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_oauth_get(76 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLCertPrivateKeyContainer() {
    $ret = $this->cloudmail_oauth_get(77 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLCertPublicKey() {
    $ret = $this->cloudmail_oauth_get(78 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_oauth_get(79 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLCertPublicKeyLength() {
    $ret = $this->cloudmail_oauth_get(80 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLCertSerialNumber() {
    $ret = $this->cloudmail_oauth_get(81 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLCertSignatureAlgorithm() {
    $ret = $this->cloudmail_oauth_get(82 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLCertStore() {
    $ret = $this->cloudmail_oauth_get(83 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertStore($value) {
    $ret = $this->cloudmail_oauth_set(83, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLCertStorePassword() {
    $ret = $this->cloudmail_oauth_get(84 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertStorePassword($value) {
    $ret = $this->cloudmail_oauth_set(84, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLCertStoreType() {
    $ret = $this->cloudmail_oauth_get(85 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLCertStoreType($value) {
    $ret = $this->cloudmail_oauth_set(85, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLCertSubjectAltNames() {
    $ret = $this->cloudmail_oauth_get(86 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintMD5() {
    $ret = $this->cloudmail_oauth_get(87 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintSHA1() {
    $ret = $this->cloudmail_oauth_get(88 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintSHA256() {
    $ret = $this->cloudmail_oauth_get(89 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLCertUsage() {
    $ret = $this->cloudmail_oauth_get(90 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLCertUsageFlags() {
    $ret = $this->cloudmail_oauth_get(91 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLCertVersion() {
    $ret = $this->cloudmail_oauth_get(92 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLCertSubject() {
    $ret = $this->cloudmail_oauth_get(93 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertSubject($value) {
    $ret = $this->cloudmail_oauth_set(93, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLCertEncoded() {
    $ret = $this->cloudmail_oauth_get(94 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertEncoded($value) {
    $ret = $this->cloudmail_oauth_set(94, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The Secure Sockets Layer/Transport Layer Security (SSL/TLS) implementation to use.
  *
  * @access   public
  */
  public function getSSLProvider() {
    $ret = $this->cloudmail_oauth_get(95 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The Secure Sockets Layer/Transport Layer Security (SSL/TLS) implementation to use.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLProvider($value) {
    $ret = $this->cloudmail_oauth_set(95, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLServerCertEffectiveDate() {
    $ret = $this->cloudmail_oauth_get(96 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLServerCertExpirationDate() {
    $ret = $this->cloudmail_oauth_get(97 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLServerCertExtendedKeyUsage() {
    $ret = $this->cloudmail_oauth_get(98 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprint() {
    $ret = $this->cloudmail_oauth_get(99 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprintSHA1() {
    $ret = $this->cloudmail_oauth_get(100 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprintSHA256() {
    $ret = $this->cloudmail_oauth_get(101 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertIssuer() {
    $ret = $this->cloudmail_oauth_get(102 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKey() {
    $ret = $this->cloudmail_oauth_get(103 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_oauth_get(104 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKeyContainer() {
    $ret = $this->cloudmail_oauth_get(105 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertPublicKey() {
    $ret = $this->cloudmail_oauth_get(106 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLServerCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_oauth_get(107 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLServerCertPublicKeyLength() {
    $ret = $this->cloudmail_oauth_get(108 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLServerCertSerialNumber() {
    $ret = $this->cloudmail_oauth_get(109 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLServerCertSignatureAlgorithm() {
    $ret = $this->cloudmail_oauth_get(110 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLServerCertStore() {
    $ret = $this->cloudmail_oauth_get(111 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLServerCertStorePassword() {
    $ret = $this->cloudmail_oauth_get(112 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLServerCertStoreType() {
    $ret = $this->cloudmail_oauth_get(113 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertSubjectAltNames() {
    $ret = $this->cloudmail_oauth_get(114 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintMD5() {
    $ret = $this->cloudmail_oauth_get(115 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintSHA1() {
    $ret = $this->cloudmail_oauth_get(116 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintSHA256() {
    $ret = $this->cloudmail_oauth_get(117 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLServerCertUsage() {
    $ret = $this->cloudmail_oauth_get(118 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertUsageFlags() {
    $ret = $this->cloudmail_oauth_get(119 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLServerCertVersion() {
    $ret = $this->cloudmail_oauth_get(120 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLServerCertSubject() {
    $ret = $this->cloudmail_oauth_get(121 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLServerCertEncoded() {
    $ret = $this->cloudmail_oauth_get(122 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The timeout for the class.
  *
  * @access   public
  */
  public function getTimeout() {
    $ret = $this->cloudmail_oauth_get(123 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The timeout for the class.
  *
  * @access   public
  * @param    int   value
  */
  public function setTimeout($value) {
    $ret = $this->cloudmail_oauth_set(123, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The contents of the last response from the server.
  *
  * @access   public
  */
  public function getTransferredData() {
    $ret = $this->cloudmail_oauth_get(124 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The full set of headers as received from the server.
  *
  * @access   public
  */
  public function getTransferredHeaders() {
    $ret = $this->cloudmail_oauth_get(125 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether Proof Key for Code Exchange (PKCE) should be used.
  *
  * @access   public
  */
  public function getUsePKCE() {
    $ret = $this->cloudmail_oauth_get(126 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether Proof Key for Code Exchange (PKCE) should be used.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setUsePKCE($value) {
    $ret = $this->cloudmail_oauth_set(126, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The local port on which the embedded web server listens.
  *
  * @access   public
  */
  public function getWebServerPort() {
    $ret = $this->cloudmail_oauth_get(127 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The local port on which the embedded web server listens.
  *
  * @access   public
  * @param    int   value
  */
  public function setWebServerPort($value) {
    $ret = $this->cloudmail_oauth_set(127, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getWebServerSSLCertEffectiveDate() {
    $ret = $this->cloudmail_oauth_get(128 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getWebServerSSLCertExpirationDate() {
    $ret = $this->cloudmail_oauth_get(129 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getWebServerSSLCertExtendedKeyUsage() {
    $ret = $this->cloudmail_oauth_get(130 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertFingerprint() {
    $ret = $this->cloudmail_oauth_get(131 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertFingerprintSHA1() {
    $ret = $this->cloudmail_oauth_get(132 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertFingerprintSHA256() {
    $ret = $this->cloudmail_oauth_get(133 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertIssuer() {
    $ret = $this->cloudmail_oauth_get(134 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getWebServerSSLCertPrivateKey() {
    $ret = $this->cloudmail_oauth_get(135 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_oauth_get(136 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getWebServerSSLCertPrivateKeyContainer() {
    $ret = $this->cloudmail_oauth_get(137 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertPublicKey() {
    $ret = $this->cloudmail_oauth_get(138 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getWebServerSSLCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_oauth_get(139 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getWebServerSSLCertPublicKeyLength() {
    $ret = $this->cloudmail_oauth_get(140 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getWebServerSSLCertSerialNumber() {
    $ret = $this->cloudmail_oauth_get(141 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getWebServerSSLCertSignatureAlgorithm() {
    $ret = $this->cloudmail_oauth_get(142 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertStore() {
    $ret = $this->cloudmail_oauth_get(143 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  * @param    string   value
  */
  public function setWebServerSSLCertStore($value) {
    $ret = $this->cloudmail_oauth_set(143, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getWebServerSSLCertStorePassword() {
    $ret = $this->cloudmail_oauth_get(144 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  * @param    string   value
  */
  public function setWebServerSSLCertStorePassword($value) {
    $ret = $this->cloudmail_oauth_set(144, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertStoreType() {
    $ret = $this->cloudmail_oauth_get(145 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  * @param    int   value
  */
  public function setWebServerSSLCertStoreType($value) {
    $ret = $this->cloudmail_oauth_set(145, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertSubjectAltNames() {
    $ret = $this->cloudmail_oauth_get(146 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertThumbprintMD5() {
    $ret = $this->cloudmail_oauth_get(147 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertThumbprintSHA1() {
    $ret = $this->cloudmail_oauth_get(148 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertThumbprintSHA256() {
    $ret = $this->cloudmail_oauth_get(149 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getWebServerSSLCertUsage() {
    $ret = $this->cloudmail_oauth_get(150 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getWebServerSSLCertUsageFlags() {
    $ret = $this->cloudmail_oauth_get(151 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getWebServerSSLCertVersion() {
    $ret = $this->cloudmail_oauth_get(152 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getWebServerSSLCertSubject() {
    $ret = $this->cloudmail_oauth_get(153 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  * @param    string   value
  */
  public function setWebServerSSLCertSubject($value) {
    $ret = $this->cloudmail_oauth_set(153, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getWebServerSSLCertEncoded() {
    $ret = $this->cloudmail_oauth_get(154 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  * @param    string   value
  */
  public function setWebServerSSLCertEncoded($value) {
    $ret = $this->cloudmail_oauth_set(154, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether the web server requires Secure Sockets Layer (SSL) connections.
  *
  * @access   public
  */
  public function getWebServerSSLEnabled() {
    $ret = $this->cloudmail_oauth_get(155 );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether the web server requires Secure Sockets Layer (SSL) connections.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setWebServerSSLEnabled($value) {
    $ret = $this->cloudmail_oauth_set(155, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }



  public function getRuntimeLicense() {
    $ret = $this->cloudmail_oauth_get(8001/*PID_RUNTIME_LICENSE*/ );
    $error_code = $this->cloudmail_oauth_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }

  public function setRuntimeLicense($value) {
    $ret = $this->cloudmail_oauth_set(8001/*PID_RUNTIME_LICENSE*/, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_oauth_get_last_error());
    }
    return $ret;
  }
  
 /**
  * Fired immediately after a connection completes (or fails).
  *
  * @access   public
  * @param    array   Array of event parameters: statuscode, description    
  */
  public function fireConnected($param) {
    return $param;
  }


 /**
  * Fired to indicate changes in the connection state.
  *
  * @access   public
  * @param    array   Array of event parameters: connectionevent, statuscode, description    
  */
  public function fireConnectionStatus($param) {
    return $param;
  }


 /**
  * Fired when a connection is closed.
  *
  * @access   public
  * @param    array   Array of event parameters: statuscode, description    
  */
  public function fireDisconnected($param) {
    return $param;
  }


 /**
  * Fired when a document finishes transferring.
  *
  * @access   public
  * @param    array   Array of event parameters: direction    
  */
  public function fireEndTransfer($param) {
    return $param;
  }


 /**
  * Fired when information is available about errors during data delivery.
  *
  * @access   public
  * @param    array   Array of event parameters: errorcode, description    
  */
  public function fireError($param) {
    return $param;
  }


 /**
  * Fired every time a header line comes in.
  *
  * @access   public
  * @param    array   Array of event parameters: field, value    
  */
  public function fireHeader($param) {
    return $param;
  }


 /**
  * Fires before launching a browser with the authorization URL.
  *
  * @access   public
  * @param    array   Array of event parameters: url, command    
  */
  public function fireLaunchBrowser($param) {
    return $param;
  }


 /**
  * Fired once for each log message.
  *
  * @access   public
  * @param    array   Array of event parameters: loglevel, message, logtype    
  */
  public function fireLog($param) {
    return $param;
  }


 /**
  * Fired when a redirection is received from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: location, accept    
  */
  public function fireRedirect($param) {
    return $param;
  }


 /**
  * Fired when the user is redirected to the embedded web server.
  *
  * @access   public
  * @param    array   Array of event parameters: urlpath, querystring, responseheaders, responsebody    
  */
  public function fireReturnURL($param) {
    return $param;
  }


 /**
  * Fired for every cookie set by the server.
  *
  * @access   public
  * @param    array   Array of event parameters: name, value, expires, domain, path, secure    
  */
  public function fireSetCookie($param) {
    return $param;
  }


 /**
  * Fired after the server presents its certificate to the client.
  *
  * @access   public
  * @param    array   Array of event parameters: certencoded, certsubject, certissuer, status, accept    
  */
  public function fireSSLServerAuthentication($param) {
    return $param;
  }


 /**
  * Fired when secure connection progress messages are available.
  *
  * @access   public
  * @param    array   Array of event parameters: message    
  */
  public function fireSSLStatus($param) {
    return $param;
  }


 /**
  * Fired when a document starts transferring (after the headers).
  *
  * @access   public
  * @param    array   Array of event parameters: direction    
  */
  public function fireStartTransfer($param) {
    return $param;
  }


 /**
  * Fired when the HTTP status line is received from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: httpversion, statuscode, description    
  */
  public function fireStatus($param) {
    return $param;
  }


 /**
  * Fired while a document transfers (delivers document).
  *
  * @access   public
  * @param    array   Array of event parameters: direction, bytestransferred, percentdone, text    
  */
  public function fireTransfer($param) {
    return $param;
  }



  private function cloudmail_oauth_open($key) {
    global $lib_cloudmail_oauth;
    global $objMap;

    $id = spl_object_id($this);
    $objMap[$id] = $this;
    $objId = $lib_cloudmail_oauth->cast("void*", $id);

    $this->handle = $lib_cloudmail_oauth->CloudMail_OAuth_Create("cloudmail_oauth_callback", $objId, null, 0);
    if (!isset($this->handle)) {
      error_log("out of memory", 0);
    }

    $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 8004/*PID_KEYCHECK_PHP*/, 0, $key, 0);
  }

  private function cloudmail_oauth_close() {
    global $lib_cloudmail_oauth;
    global $objMap;

    if (isset($this->handle)) {
      $lib_cloudmail_oauth->CloudMail_OAuth_Destroy($this->handle);
      $this->handle = null;
    }
    
    $objMap[spl_object_id($this)] = null;
  }

  private function cloudmail_oauth_get_last_error() {
    global $lib_cloudmail_oauth;
    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_GetLastError($this->handle);
    if (isset($retval)) return FFI::string($lib_cloudmail_oauth->cast("char*", $retval));
  }

  private function cloudmail_oauth_get_last_error_code() {
    global $lib_cloudmail_oauth;
    return (int)$lib_cloudmail_oauth->CloudMail_OAuth_GetLastErrorCode($this->handle);
  }
  

 /**
  * Adds a cookie and the corresponding value to the outgoing request headers.
  *
  * @access   private
  * @param    string    cookiename

  * @param    string    cookievalue

  */
  private function cloudmail_oauth_do_addcookie($cookiename, $cookievalue) {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $cookiename . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $cookievalue . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 2, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Adds a name-value pair to the query string parameters of the outgoing request.
  *
  * @access   private
  * @param    string    paramname

  * @param    string    paramvalue

  */
  private function cloudmail_oauth_do_addparam($paramname, $paramvalue) {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $paramname . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $paramvalue . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 3, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Sets or retrieves a configuration setting.
  *
  * @access   private
  * @param    string    configurationstring

  */
  private function cloudmail_oauth_do_config($configurationstring) {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $configurationstring . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 4, 1, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * This method processes events from the internal message queue.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_doevents() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 5, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Gets the authorization string required to access the protected resource.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_getauthorization() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 6, 0, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Builds and returns the URL to which the user should be redirected for authorization.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_getauthorizationurl() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 7, 0, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_oauth->cast("char*", $param[0]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Gets a specific parameter from a query string.
  *
  * @access   private
  * @param    string    paramname

  */
  private function cloudmail_oauth_do_getparam($paramname) {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $paramname . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_oauth->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 8, 1, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_oauth->cast("char*", $param[1]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * This method interrupts the current method.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_interrupt() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 9, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * This method will reset the class.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_reset() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 10, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Starts the embedded web server.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_startwebserver() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 11, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Stops the embedded web server.
  *
  * @access   private
  */
  private function cloudmail_oauth_do_stopwebserver() {
    global $lib_cloudmail_oauth;
    $param = $lib_cloudmail_oauth->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_oauth->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Do($this->handle, 12, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

  
  private function cloudmail_oauth_get($pid, $index = 0) {
    global $lib_cloudmail_oauth;

    switch ($pid) {
      case 0:
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 0, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
      case 1: { /* ACCESSTOKEN */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 1, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 2: { /* ACCESSTOKENEXP */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 2, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 3: { /* AUTHORIZATIONCODE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 3, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 4: { /* AUTHORIZATIONSCOPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 4, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 5: { /* CLIENTID */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 5, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 6: { /* CLIENTPROFILE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 6, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 7: { /* CLIENTSECRET */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 7, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 8: { /* CONNECTED */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 8, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 9: { /* COOKIECOUNT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 9, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 10: { /* COOKIEDOMAIN */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 10, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 11: { /* COOKIEEXPIRATION */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 11, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 12: { /* COOKIENAME */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 12, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 13: { /* COOKIEPATH */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 13, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 14: { /* COOKIESECURE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 14, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 15: { /* COOKIEVALUE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 15, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 16: { /* FIREWALLAUTODETECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 16, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 17: { /* FIREWALLTYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 17, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 18: { /* FIREWALLHOST */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 18, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 19: { /* FIREWALLPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 19, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 20: { /* FIREWALLPORT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 20, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 21: { /* FIREWALLUSER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 21, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 22: { /* FOLLOWREDIRECTS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 22, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 23: { /* GRANTTYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 23, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 24: { /* IDLE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 24, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 25: { /* LOCALHOST */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 25, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 26: { /* OTHERHEADERS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 26, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 27: { /* PARAMCOUNT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 27, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 28: { /* PARAMNAME */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 28, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 29: { /* PARAMVALUE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 29, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 30: { /* PROXYAUTHSCHEME */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 30, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 31: { /* PROXYAUTODETECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 31, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 32: { /* PROXYPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 32, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 33: { /* PROXYPORT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 33, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 34: { /* PROXYSERVER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 34, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 35: { /* PROXYSSL */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 35, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 36: { /* PROXYUSER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 36, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 37: { /* REFRESHTOKEN */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 37, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 38: { /* RETURNURL */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 38, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 39: { /* SERVERAUTHURL */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 39, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 40: { /* SERVERTOKENURL */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 40, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 41: { /* SSLACCEPTSERVERCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 41, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 42: { /* SSLACCEPTSERVERCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 42, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 43: { /* SSLACCEPTSERVERCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 43, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 44: { /* SSLACCEPTSERVERCERTFINGERPRINT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 44, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 45: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 45, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 46: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 46, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 47: { /* SSLACCEPTSERVERCERTISSUER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 47, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 48: { /* SSLACCEPTSERVERCERTPRIVATEKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 48, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 49: { /* SSLACCEPTSERVERCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 49, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 50: { /* SSLACCEPTSERVERCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 50, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 51: { /* SSLACCEPTSERVERCERTPUBLICKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 51, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 52: { /* SSLACCEPTSERVERCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 52, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 53: { /* SSLACCEPTSERVERCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 53, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 54: { /* SSLACCEPTSERVERCERTSERIALNUMBER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 54, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 55: { /* SSLACCEPTSERVERCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 55, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 56: { /* SSLACCEPTSERVERCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 56, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 57: { /* SSLACCEPTSERVERCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 57, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 58: { /* SSLACCEPTSERVERCERTSTORETYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 58, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 59: { /* SSLACCEPTSERVERCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 59, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 60: { /* SSLACCEPTSERVERCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 60, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 61: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 61, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 62: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 62, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 63: { /* SSLACCEPTSERVERCERTUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 63, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 64: { /* SSLACCEPTSERVERCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 64, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 65: { /* SSLACCEPTSERVERCERTVERSION */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 65, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 66: { /* SSLACCEPTSERVERCERTSUBJECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 66, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 67: { /* SSLACCEPTSERVERCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 67, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 68: { /* SSLCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 68, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 69: { /* SSLCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 69, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 70: { /* SSLCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 70, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 71: { /* SSLCERTFINGERPRINT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 71, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 72: { /* SSLCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 72, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 73: { /* SSLCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 73, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 74: { /* SSLCERTISSUER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 74, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 75: { /* SSLCERTPRIVATEKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 75, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 76: { /* SSLCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 76, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 77: { /* SSLCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 77, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 78: { /* SSLCERTPUBLICKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 78, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 79: { /* SSLCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 79, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 80: { /* SSLCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 80, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 81: { /* SSLCERTSERIALNUMBER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 81, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 82: { /* SSLCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 82, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 83: { /* SSLCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 83, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 84: { /* SSLCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 84, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 85: { /* SSLCERTSTORETYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 85, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 86: { /* SSLCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 86, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 87: { /* SSLCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 87, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 88: { /* SSLCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 88, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 89: { /* SSLCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 89, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 90: { /* SSLCERTUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 90, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 91: { /* SSLCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 91, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 92: { /* SSLCERTVERSION */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 92, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 93: { /* SSLCERTSUBJECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 93, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 94: { /* SSLCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 94, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 95: { /* SSLPROVIDER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 95, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 96: { /* SSLSERVERCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 96, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 97: { /* SSLSERVERCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 97, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 98: { /* SSLSERVERCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 98, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 99: { /* SSLSERVERCERTFINGERPRINT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 99, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 100: { /* SSLSERVERCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 100, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 101: { /* SSLSERVERCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 101, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 102: { /* SSLSERVERCERTISSUER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 102, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 103: { /* SSLSERVERCERTPRIVATEKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 103, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 104: { /* SSLSERVERCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 104, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 105: { /* SSLSERVERCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 105, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 106: { /* SSLSERVERCERTPUBLICKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 106, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 107: { /* SSLSERVERCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 107, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 108: { /* SSLSERVERCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 108, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 109: { /* SSLSERVERCERTSERIALNUMBER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 109, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 110: { /* SSLSERVERCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 110, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 111: { /* SSLSERVERCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 111, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 112: { /* SSLSERVERCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 112, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 113: { /* SSLSERVERCERTSTORETYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 113, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 114: { /* SSLSERVERCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 114, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 115: { /* SSLSERVERCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 115, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 116: { /* SSLSERVERCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 116, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 117: { /* SSLSERVERCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 117, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 118: { /* SSLSERVERCERTUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 118, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 119: { /* SSLSERVERCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 119, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 120: { /* SSLSERVERCERTVERSION */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 120, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 121: { /* SSLSERVERCERTSUBJECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 121, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 122: { /* SSLSERVERCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 122, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 123: { /* TIMEOUT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 123, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 124: { /* TRANSFERREDDATA */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 124, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 125: { /* TRANSFERREDHEADERS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 125, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 126: { /* USEPKCE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 126, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 127: { /* WEBSERVERPORT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 127, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 128: { /* WEBSERVERSSLCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 128, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 129: { /* WEBSERVERSSLCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 129, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 130: { /* WEBSERVERSSLCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 130, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 131: { /* WEBSERVERSSLCERTFINGERPRINT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 131, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 132: { /* WEBSERVERSSLCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 132, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 133: { /* WEBSERVERSSLCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 133, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 134: { /* WEBSERVERSSLCERTISSUER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 134, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 135: { /* WEBSERVERSSLCERTPRIVATEKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 135, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 136: { /* WEBSERVERSSLCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 136, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 137: { /* WEBSERVERSSLCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 137, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 138: { /* WEBSERVERSSLCERTPUBLICKEY */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 138, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 139: { /* WEBSERVERSSLCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 139, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 140: { /* WEBSERVERSSLCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 140, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 141: { /* WEBSERVERSSLCERTSERIALNUMBER */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 141, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 142: { /* WEBSERVERSSLCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 142, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 143: { /* WEBSERVERSSLCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 143, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 144: { /* WEBSERVERSSLCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 144, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 145: { /* WEBSERVERSSLCERTSTORETYPE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 145, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 146: { /* WEBSERVERSSLCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 146, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 147: { /* WEBSERVERSSLCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 147, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 148: { /* WEBSERVERSSLCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 148, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 149: { /* WEBSERVERSSLCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 149, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 150: { /* WEBSERVERSSLCERTUSAGE */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 150, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 151: { /* WEBSERVERSSLCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 151, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_oauth->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 152: { /* WEBSERVERSSLCERTVERSION */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 152, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 153: { /* WEBSERVERSSLCERTSUBJECT */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 153, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }

      case 154: { /* WEBSERVERSSLCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_oauth->new("int"));
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 154, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret), $blen[0]);
        break;
      }

      case 155: { /* WEBSERVERSSLENABLED */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 155, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 8001/*PID_RUNTIME_LICENSE*/: { /* RuntimeLicense */
        $ret = $lib_cloudmail_oauth->CloudMail_OAuth_Get($this->handle, 8001, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_oauth->cast("char*", $ret));
        break;
      }
      default: {
        php_error(E_ERROR, "invalid property id");
        break;
      }
    }

    return -1;
  }

  private function cloudmail_oauth_set($pid, $value, $index = 0) {
    global $lib_cloudmail_oauth;
    $retval = -1;

    switch ($pid) {

      case 1: { /* ACCESSTOKEN */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 1, (int)$index, (string)$value, 0);
        break;
      }


      case 2: { /* ACCESSTOKENEXP */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 2, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for AccessTokenExp.", 0);
        }
        break;
      }


      case 3: { /* AUTHORIZATIONCODE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 3, (int)$index, (string)$value, 0);
        break;
      }


      case 4: { /* AUTHORIZATIONSCOPE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 4, (int)$index, (string)$value, 0);
        break;
      }


      case 5: { /* CLIENTID */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 5, (int)$index, (string)$value, 0);
        break;
      }


      case 6: { /* CLIENTPROFILE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 6, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ClientProfile.", 0);
        }
        break;
      }


      case 7: { /* CLIENTSECRET */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 7, (int)$index, (string)$value, 0);
        break;
      }


      case 8: { /* CONNECTED */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 8, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for Connected.", 0);
        }
        break;
      }


      case 9: { /* COOKIECOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 9, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for CookieCount.", 0);
        }
        break;
      }


      case 10: { /* COOKIEDOMAIN */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 10, (int)$index, (string)$value, 0);
        break;
      }


      case 11: { /* COOKIEEXPIRATION */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 11, (int)$index, (string)$value, 0);
        break;
      }


      case 12: { /* COOKIENAME */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 12, (int)$index, (string)$value, 0);
        break;
      }


      case 13: { /* COOKIEPATH */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 13, (int)$index, (string)$value, 0);
        break;
      }


      case 14: { /* COOKIESECURE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 14, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for CookieSecure.", 0);
        }
        break;
      }


      case 15: { /* COOKIEVALUE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 15, (int)$index, (string)$value, 0);
        break;
      }


      case 16: { /* FIREWALLAUTODETECT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 16, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallAutoDetect.", 0);
        }
        break;
      }


      case 17: { /* FIREWALLTYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 17, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallType.", 0);
        }
        break;
      }


      case 18: { /* FIREWALLHOST */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 18, (int)$index, (string)$value, 0);
        break;
      }


      case 19: { /* FIREWALLPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 19, (int)$index, (string)$value, 0);
        break;
      }


      case 20: { /* FIREWALLPORT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 20, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallPort.", 0);
        }
        break;
      }


      case 21: { /* FIREWALLUSER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 21, (int)$index, (string)$value, 0);
        break;
      }


      case 22: { /* FOLLOWREDIRECTS */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 22, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FollowRedirects.", 0);
        }
        break;
      }


      case 23: { /* GRANTTYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 23, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for GrantType.", 0);
        }
        break;
      }


      case 24: { /* IDLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 24, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for Idle.", 0);
        }
        break;
      }


      case 25: { /* LOCALHOST */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 25, (int)$index, (string)$value, 0);
        break;
      }


      case 26: { /* OTHERHEADERS */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 26, (int)$index, (string)$value, 0);
        break;
      }


      case 27: { /* PARAMCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 27, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ParamCount.", 0);
        }
        break;
      }


      case 28: { /* PARAMNAME */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 28, (int)$index, (string)$value, 0);
        break;
      }


      case 29: { /* PARAMVALUE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 29, (int)$index, (string)$value, 0);
        break;
      }


      case 30: { /* PROXYAUTHSCHEME */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 30, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyAuthScheme.", 0);
        }
        break;
      }


      case 31: { /* PROXYAUTODETECT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 31, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyAutoDetect.", 0);
        }
        break;
      }


      case 32: { /* PROXYPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 32, (int)$index, (string)$value, 0);
        break;
      }


      case 33: { /* PROXYPORT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 33, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyPort.", 0);
        }
        break;
      }


      case 34: { /* PROXYSERVER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 34, (int)$index, (string)$value, 0);
        break;
      }


      case 35: { /* PROXYSSL */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 35, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxySSL.", 0);
        }
        break;
      }


      case 36: { /* PROXYUSER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 36, (int)$index, (string)$value, 0);
        break;
      }


      case 37: { /* REFRESHTOKEN */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 37, (int)$index, (string)$value, 0);
        break;
      }


      case 38: { /* RETURNURL */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 38, (int)$index, (string)$value, 0);
        break;
      }


      case 39: { /* SERVERAUTHURL */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 39, (int)$index, (string)$value, 0);
        break;
      }


      case 40: { /* SERVERTOKENURL */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 40, (int)$index, (string)$value, 0);
        break;
      }


      case 41: { /* SSLACCEPTSERVERCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 41, (int)$index, (string)$value, 0);
        break;
      }


      case 42: { /* SSLACCEPTSERVERCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 42, (int)$index, (string)$value, 0);
        break;
      }


      case 43: { /* SSLACCEPTSERVERCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 43, (int)$index, (string)$value, 0);
        break;
      }


      case 44: { /* SSLACCEPTSERVERCERTFINGERPRINT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 44, (int)$index, (string)$value, 0);
        break;
      }


      case 45: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 45, (int)$index, (string)$value, 0);
        break;
      }


      case 46: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 46, (int)$index, (string)$value, 0);
        break;
      }


      case 47: { /* SSLACCEPTSERVERCERTISSUER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 47, (int)$index, (string)$value, 0);
        break;
      }


      case 48: { /* SSLACCEPTSERVERCERTPRIVATEKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 48, (int)$index, (string)$value, 0);
        break;
      }


      case 49: { /* SSLACCEPTSERVERCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 49, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 50: { /* SSLACCEPTSERVERCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 50, (int)$index, (string)$value, 0);
        break;
      }


      case 51: { /* SSLACCEPTSERVERCERTPUBLICKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 51, (int)$index, (string)$value, 0);
        break;
      }


      case 52: { /* SSLACCEPTSERVERCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 52, (int)$index, (string)$value, 0);
        break;
      }


      case 53: { /* SSLACCEPTSERVERCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 53, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertPublicKeyLength.", 0);
        }
        break;
      }


      case 54: { /* SSLACCEPTSERVERCERTSERIALNUMBER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 54, (int)$index, (string)$value, 0);
        break;
      }


      case 55: { /* SSLACCEPTSERVERCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 55, (int)$index, (string)$value, 0);
        break;
      }


      case 56: { /* SSLACCEPTSERVERCERTSTORE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 56, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 57: { /* SSLACCEPTSERVERCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 57, (int)$index, (string)$value, 0);
        break;
      }


      case 58: { /* SSLACCEPTSERVERCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 58, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertStoreType.", 0);
        }
        break;
      }


      case 59: { /* SSLACCEPTSERVERCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 59, (int)$index, (string)$value, 0);
        break;
      }


      case 60: { /* SSLACCEPTSERVERCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 60, (int)$index, (string)$value, 0);
        break;
      }


      case 61: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 61, (int)$index, (string)$value, 0);
        break;
      }


      case 62: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 62, (int)$index, (string)$value, 0);
        break;
      }


      case 63: { /* SSLACCEPTSERVERCERTUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 63, (int)$index, (string)$value, 0);
        break;
      }


      case 64: { /* SSLACCEPTSERVERCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 64, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertUsageFlags.", 0);
        }
        break;
      }


      case 65: { /* SSLACCEPTSERVERCERTVERSION */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 65, (int)$index, (string)$value, 0);
        break;
      }


      case 66: { /* SSLACCEPTSERVERCERTSUBJECT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 66, (int)$index, (string)$value, 0);
        break;
      }


      case 67: { /* SSLACCEPTSERVERCERTENCODED */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 67, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 68: { /* SSLCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 68, (int)$index, (string)$value, 0);
        break;
      }


      case 69: { /* SSLCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 69, (int)$index, (string)$value, 0);
        break;
      }


      case 70: { /* SSLCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 70, (int)$index, (string)$value, 0);
        break;
      }


      case 71: { /* SSLCERTFINGERPRINT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 71, (int)$index, (string)$value, 0);
        break;
      }


      case 72: { /* SSLCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 72, (int)$index, (string)$value, 0);
        break;
      }


      case 73: { /* SSLCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 73, (int)$index, (string)$value, 0);
        break;
      }


      case 74: { /* SSLCERTISSUER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 74, (int)$index, (string)$value, 0);
        break;
      }


      case 75: { /* SSLCERTPRIVATEKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 75, (int)$index, (string)$value, 0);
        break;
      }


      case 76: { /* SSLCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 76, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 77: { /* SSLCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 77, (int)$index, (string)$value, 0);
        break;
      }


      case 78: { /* SSLCERTPUBLICKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 78, (int)$index, (string)$value, 0);
        break;
      }


      case 79: { /* SSLCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 79, (int)$index, (string)$value, 0);
        break;
      }


      case 80: { /* SSLCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 80, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertPublicKeyLength.", 0);
        }
        break;
      }


      case 81: { /* SSLCERTSERIALNUMBER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 81, (int)$index, (string)$value, 0);
        break;
      }


      case 82: { /* SSLCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 82, (int)$index, (string)$value, 0);
        break;
      }


      case 83: { /* SSLCERTSTORE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 83, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 84: { /* SSLCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 84, (int)$index, (string)$value, 0);
        break;
      }


      case 85: { /* SSLCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 85, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertStoreType.", 0);
        }
        break;
      }


      case 86: { /* SSLCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 86, (int)$index, (string)$value, 0);
        break;
      }


      case 87: { /* SSLCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 87, (int)$index, (string)$value, 0);
        break;
      }


      case 88: { /* SSLCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 88, (int)$index, (string)$value, 0);
        break;
      }


      case 89: { /* SSLCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 89, (int)$index, (string)$value, 0);
        break;
      }


      case 90: { /* SSLCERTUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 90, (int)$index, (string)$value, 0);
        break;
      }


      case 91: { /* SSLCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 91, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertUsageFlags.", 0);
        }
        break;
      }


      case 92: { /* SSLCERTVERSION */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 92, (int)$index, (string)$value, 0);
        break;
      }


      case 93: { /* SSLCERTSUBJECT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 93, (int)$index, (string)$value, 0);
        break;
      }


      case 94: { /* SSLCERTENCODED */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 94, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 95: { /* SSLPROVIDER */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 95, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLProvider.", 0);
        }
        break;
      }


      case 96: { /* SSLSERVERCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 96, (int)$index, (string)$value, 0);
        break;
      }


      case 97: { /* SSLSERVERCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 97, (int)$index, (string)$value, 0);
        break;
      }


      case 98: { /* SSLSERVERCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 98, (int)$index, (string)$value, 0);
        break;
      }


      case 99: { /* SSLSERVERCERTFINGERPRINT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 99, (int)$index, (string)$value, 0);
        break;
      }


      case 100: { /* SSLSERVERCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 100, (int)$index, (string)$value, 0);
        break;
      }


      case 101: { /* SSLSERVERCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 101, (int)$index, (string)$value, 0);
        break;
      }


      case 102: { /* SSLSERVERCERTISSUER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 102, (int)$index, (string)$value, 0);
        break;
      }


      case 103: { /* SSLSERVERCERTPRIVATEKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 103, (int)$index, (string)$value, 0);
        break;
      }


      case 104: { /* SSLSERVERCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 104, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 105: { /* SSLSERVERCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 105, (int)$index, (string)$value, 0);
        break;
      }


      case 106: { /* SSLSERVERCERTPUBLICKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 106, (int)$index, (string)$value, 0);
        break;
      }


      case 107: { /* SSLSERVERCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 107, (int)$index, (string)$value, 0);
        break;
      }


      case 108: { /* SSLSERVERCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 108, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertPublicKeyLength.", 0);
        }
        break;
      }


      case 109: { /* SSLSERVERCERTSERIALNUMBER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 109, (int)$index, (string)$value, 0);
        break;
      }


      case 110: { /* SSLSERVERCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 110, (int)$index, (string)$value, 0);
        break;
      }


      case 111: { /* SSLSERVERCERTSTORE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 111, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 112: { /* SSLSERVERCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 112, (int)$index, (string)$value, 0);
        break;
      }


      case 113: { /* SSLSERVERCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 113, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertStoreType.", 0);
        }
        break;
      }


      case 114: { /* SSLSERVERCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 114, (int)$index, (string)$value, 0);
        break;
      }


      case 115: { /* SSLSERVERCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 115, (int)$index, (string)$value, 0);
        break;
      }


      case 116: { /* SSLSERVERCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 116, (int)$index, (string)$value, 0);
        break;
      }


      case 117: { /* SSLSERVERCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 117, (int)$index, (string)$value, 0);
        break;
      }


      case 118: { /* SSLSERVERCERTUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 118, (int)$index, (string)$value, 0);
        break;
      }


      case 119: { /* SSLSERVERCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 119, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertUsageFlags.", 0);
        }
        break;
      }


      case 120: { /* SSLSERVERCERTVERSION */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 120, (int)$index, (string)$value, 0);
        break;
      }


      case 121: { /* SSLSERVERCERTSUBJECT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 121, (int)$index, (string)$value, 0);
        break;
      }


      case 122: { /* SSLSERVERCERTENCODED */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 122, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 123: { /* TIMEOUT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 123, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for Timeout.", 0);
        }
        break;
      }


      case 124: { /* TRANSFERREDDATA */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 124, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 125: { /* TRANSFERREDHEADERS */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 125, (int)$index, (string)$value, 0);
        break;
      }


      case 126: { /* USEPKCE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 126, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for UsePKCE.", 0);
        }
        break;
      }


      case 127: { /* WEBSERVERPORT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 127, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerPort.", 0);
        }
        break;
      }


      case 128: { /* WEBSERVERSSLCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 128, (int)$index, (string)$value, 0);
        break;
      }


      case 129: { /* WEBSERVERSSLCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 129, (int)$index, (string)$value, 0);
        break;
      }


      case 130: { /* WEBSERVERSSLCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 130, (int)$index, (string)$value, 0);
        break;
      }


      case 131: { /* WEBSERVERSSLCERTFINGERPRINT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 131, (int)$index, (string)$value, 0);
        break;
      }


      case 132: { /* WEBSERVERSSLCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 132, (int)$index, (string)$value, 0);
        break;
      }


      case 133: { /* WEBSERVERSSLCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 133, (int)$index, (string)$value, 0);
        break;
      }


      case 134: { /* WEBSERVERSSLCERTISSUER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 134, (int)$index, (string)$value, 0);
        break;
      }


      case 135: { /* WEBSERVERSSLCERTPRIVATEKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 135, (int)$index, (string)$value, 0);
        break;
      }


      case 136: { /* WEBSERVERSSLCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 136, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerSSLCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 137: { /* WEBSERVERSSLCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 137, (int)$index, (string)$value, 0);
        break;
      }


      case 138: { /* WEBSERVERSSLCERTPUBLICKEY */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 138, (int)$index, (string)$value, 0);
        break;
      }


      case 139: { /* WEBSERVERSSLCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 139, (int)$index, (string)$value, 0);
        break;
      }


      case 140: { /* WEBSERVERSSLCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 140, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerSSLCertPublicKeyLength.", 0);
        }
        break;
      }


      case 141: { /* WEBSERVERSSLCERTSERIALNUMBER */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 141, (int)$index, (string)$value, 0);
        break;
      }


      case 142: { /* WEBSERVERSSLCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 142, (int)$index, (string)$value, 0);
        break;
      }


      case 143: { /* WEBSERVERSSLCERTSTORE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 143, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 144: { /* WEBSERVERSSLCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 144, (int)$index, (string)$value, 0);
        break;
      }


      case 145: { /* WEBSERVERSSLCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 145, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerSSLCertStoreType.", 0);
        }
        break;
      }


      case 146: { /* WEBSERVERSSLCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 146, (int)$index, (string)$value, 0);
        break;
      }


      case 147: { /* WEBSERVERSSLCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 147, (int)$index, (string)$value, 0);
        break;
      }


      case 148: { /* WEBSERVERSSLCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 148, (int)$index, (string)$value, 0);
        break;
      }


      case 149: { /* WEBSERVERSSLCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 149, (int)$index, (string)$value, 0);
        break;
      }


      case 150: { /* WEBSERVERSSLCERTUSAGE */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 150, (int)$index, (string)$value, 0);
        break;
      }


      case 151: { /* WEBSERVERSSLCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 151, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerSSLCertUsageFlags.", 0);
        }
        break;
      }


      case 152: { /* WEBSERVERSSLCERTVERSION */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 152, (int)$index, (string)$value, 0);
        break;
      }


      case 153: { /* WEBSERVERSSLCERTSUBJECT */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 153, (int)$index, (string)$value, 0);
        break;
      }


      case 154: { /* WEBSERVERSSLCERTENCODED */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 154, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 155: { /* WEBSERVERSSLENABLED */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 155, (int)$index,
              $lib_cloudmail_oauth->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for WebServerSSLEnabled.", 0);
        }
        break;
      }

      case 8001/*PID_RUNTIME_LICENSE*/: { /* RuntimeLicense */
        $retval = $lib_cloudmail_oauth->CloudMail_OAuth_Set($this->handle, 8001/*PID_RUNTIME_LICENSE*/, (int)$index, (string)$value, 0);
        break;
      }
      default: {
        error_log("Invalid property id", 0);
        break;
      }
    }
    return $retval;
  }
}


?>

