<?php

require_once('cloudmail_keys.php');

/**
 * Cloud Mail 2024 PHP Edition - Office365 Component
 *
 * Copyright (c) 2025 /n software inc.
 *
 * For more information, please visit http://www.nsoftware.com/.
 *
 */

global $lib_cloudmail_office365;
global $objMap;

$lib_cloudmail_office365 = FFI::cdef("
  typedef int (*CloudMail_Office365_callback) (void *pobj, int event_id, int cparam, void *param[], int cbparam[]);
  void* CloudMail_Office365_Create(CloudMail_Office365_callback lpSink, void *lpContext, char *lpOemKey, int opts);
  int   CloudMail_Office365_Destroy(void *lpObj);
  int   CloudMail_Office365_CheckIndex(void *lpObj, int propid, int arridx);
  void* CloudMail_Office365_Get(void *lpObj, int propid, int arridx, int *lpcbVal, int64_t *lpllVal);
  int   CloudMail_Office365_Set(void *lpObj, int propid, int arridx, const void *val, int cbVal);
  int   CloudMail_Office365_Do(void *lpObj, int methid, int cparam, void *param[], int cbparam[], int64_t *lpllVal);
  char* CloudMail_Office365_GetLastError(void *lpObj);
  int   CloudMail_Office365_GetLastErrorCode(void *lpObj);
  void* CloudMail_EvtStr(void *lpEvtStr, int id, void *val, int opt);",
  "libcloudmail.so.24.0");

$objMap = [];

function cloudmail_office365_callback($lpObj, $event_id, $cparam, $param, $cbparam) {
  global $lib_cloudmail_office365;
  global $objMap;
  if (FFI::isnull($lpObj)) return -1;

  $objId = $lpObj - $lib_cloudmail_office365->cast("void*", 0);
  $obj = $objMap[$objId];
  if (!isset($obj)) throw new Exception("invalid php object id");

  switch ($event_id) {
    case 1: { /* ATTACHMENTLIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["name"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["contenttype"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["contentlocation"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[3]));
        }
        if (!FFI::isnull($param[4])) {
          $param_array["contentbytes"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[4]), (int)$cbparam[4]);
        }
        if (!FFI::isnull($param[5])) {
          $param_array["contentid"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[5]));
        }
        if (!FFI::isnull($param[6])) {
          $param_array["lastmodifieddatetime"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[6]));
        }
        $param_array["isinline"] = !FFI::isnull($param[7]);
        $param_array["size"] = 0;
        if (!FFI::isnull($param[8])) {
          $param_array["size"] = $param[8] - $lib_cloudmail_office365->cast("void*", 0);
        }
      }

      $retval = $obj->fireAttachmentList($param_array);

 
      break;
    }

    case 2: { /* CATEGORYLIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["displayname"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["color"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireCategoryList($param_array);

 
      break;
    }

    case 3: { /* CONTACTFOLDERLIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["name"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["parentfolderid"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireContactFolderList($param_array);

 
      break;
    }

    case 4: { /* CONTACTLIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["parentid"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["displayname"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["firstemail"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[3]));
        }
      }

      $retval = $obj->fireContactList($param_array);

 
      break;
    }

    case 5: { /* ERROR */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["errorcode"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["errorcode"] = $param[0] - $lib_cloudmail_office365->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["description"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
      }

      $retval = $obj->fireError($param_array);

 
      break;
    }

    case 6: { /* FOLDERLIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["displayname"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["parentfolderid"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
        $param_array["childfoldercount"] = 0;
        if (!FFI::isnull($param[3])) {
          $param_array["childfoldercount"] = $param[3] - $lib_cloudmail_office365->cast("void*", 0);
        }
        $param_array["unreaditemcount"] = 0;
        if (!FFI::isnull($param[4])) {
          $param_array["unreaditemcount"] = $param[4] - $lib_cloudmail_office365->cast("void*", 0);
        }
        $param_array["totalitemcount"] = 0;
        if (!FFI::isnull($param[5])) {
          $param_array["totalitemcount"] = $param[5] - $lib_cloudmail_office365->cast("void*", 0);
        }
      }

      $retval = $obj->fireFolderList($param_array);

 
      break;
    }

    case 7: { /* LOG */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["loglevel"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["loglevel"] = $param[0] - $lib_cloudmail_office365->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["message"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["logtype"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
      }

      $retval = $obj->fireLog($param_array);

 
      break;
    }

    case 8: { /* MESSAGELIST */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["id"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
        if (!FFI::isnull($param[1])) {
          $param_array["subject"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["messagesender"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["receiveddatetime"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[3]));
        }
        if (!FFI::isnull($param[4])) {
          $param_array["sentdatetime"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[4]));
        }
        $param_array["hasattachments"] = !FFI::isnull($param[5]);
        if (!FFI::isnull($param[6])) {
          $param_array["bodycontenttype"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[6]));
        }
        if (!FFI::isnull($param[7])) {
          $param_array["bodycontent"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[7]));
        }
        if (!FFI::isnull($param[8])) {
          $param_array["bodypreview"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[8]));
        }
      }

      $retval = $obj->fireMessageList($param_array);

 
      break;
    }

    case 9: { /* SSLSERVERAUTHENTICATION */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["certencoded"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]), (int)$cbparam[0]);
        }
        if (!FFI::isnull($param[1])) {
          $param_array["certsubject"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));
        }
        if (!FFI::isnull($param[2])) {
          $param_array["certissuer"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));
        }
        if (!FFI::isnull($param[3])) {
          $param_array["status"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[3]));
        }
        $param_array["accept"] = !FFI::isnull($param[4]);
      }

      $retval = $obj->fireSSLServerAuthentication($param_array);

      if (isset($retval["accept"])) {
        $param[4] = $lib_cloudmail_office365->cast("void*", (int)$retval["accept"]);
      }
 
      break;
    }

    case 10: { /* SSLSTATUS */
      $param_array = [];
      if (!FFI::isnull($param)) {
        if (!FFI::isnull($param[0])) {
          $param_array["message"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[0]));
        }
      }

      $retval = $obj->fireSSLStatus($param_array);

 
      break;
    }

    case 11: { /* TRANSFER */
      $param_array = [];
      if (!FFI::isnull($param)) {
        $param_array["direction"] = 0;
        if (!FFI::isnull($param[0])) {
          $param_array["direction"] = $param[0] - $lib_cloudmail_office365->cast("void*", 0);
        }
        if (!FFI::isnull($param[1])) {
          $retLong64Val = $lib_cloudmail_office365->cast("int64_t*", $param[1]);
          $param_array["bytestransferred"] = (string)$retLong64Val[0];
        }
        $param_array["percentdone"] = 0;
        if (!FFI::isnull($param[2])) {
          $param_array["percentdone"] = $param[2] - $lib_cloudmail_office365->cast("void*", 0);
        }
        if (!FFI::isnull($param[3])) {
          $param_array["text"] = FFI::string($lib_cloudmail_office365->cast("char*", $param[3]), (int)$cbparam[3]);
        }
      }

      $retval = $obj->fireTransfer($param_array);

 
      break;
    }

  }

  return 0;
}

class CloudMail_Office365 {
  
  var $handle;

  public function __construct() {
    $this->cloudmail_office365_open(CLOUDMAIL_OEMKEY_71);
  }

  public function __destruct() {
    $this->cloudmail_office365_close();
  }


 /**
  * Returns the last error code.
  *
  * @access   public
  */
  public function lastError() {
    return $this->cloudmail_office365_get_last_error();
  }

  
 /**
  * Returns the last error message.
  *
  * @access   public
  */
  public function lastErrorCode() {
    return $this->cloudmail_office365_get_last_error_code();
  }


 /**
  * Adds a file attachment to an existing message.
  *
  * @access   public
  * @param    string    messageid

  * @param    string    name

  * @param    string    localfile

  */
  public function doAddAttachment($messageid, $name, $localfile) {
    $ret = $this->cloudmail_office365_do_addattachment($messageid, $name, $localfile);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Adds an item attachment to an existing message.
  *
  * @access   public
  * @param    string    messageid

  * @param    string    msgtoattachid

  * @param    string    itemjson

  */
  public function doAddItemAttachment($messageid, $msgtoattachid, $itemjson) {
    $ret = $this->cloudmail_office365_do_additemattachment($messageid, $msgtoattachid, $itemjson);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Get the authorization string required to access the protected resource.
  *
  * @access   public
  */
  public function doAuthorize() {
    $ret = $this->cloudmail_office365_do_authorize();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Sets or retrieves a configuration setting.
  *
  * @access   public
  * @param    string    configurationstring

  */
  public function doConfig($configurationstring) {
    $ret = $this->cloudmail_office365_do_config($configurationstring);
    $err = $this->cloudmail_office365_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Creates a copy of a message.
  *
  * @access   public
  * @param    string    id

  * @param    string    destinationid

  */
  public function doCopy($id, $destinationid) {
    $ret = $this->cloudmail_office365_do_copy($id, $destinationid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Copies a folder.
  *
  * @access   public
  * @param    string    id

  * @param    string    destfolderid

  */
  public function doCopyFolder($id, $destfolderid) {
    $ret = $this->cloudmail_office365_do_copyfolder($id, $destfolderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Creates a new category.
  *
  * @access   public
  * @param    string    displayname

  * @param    string    color

  */
  public function doCreateCategory($displayname, $color) {
    $ret = $this->cloudmail_office365_do_createcategory($displayname, $color);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Creates a contact.
  *
  * @access   public
  * @param    string    firstname

  * @param    string    lastname

  * @param    string    emailaddress

  * @param    string    parentid

  */
  public function doCreateContact($firstname, $lastname, $emailaddress, $parentid) {
    $ret = $this->cloudmail_office365_do_createcontact($firstname, $lastname, $emailaddress, $parentid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Creates a new email draft.
  *
  * @access   public
  * @param    int    drafttype

  * @param    string    id

  */
  public function doCreateDraft($drafttype, $id) {
    $ret = $this->cloudmail_office365_do_createdraft($drafttype, $id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Creates a new folder.
  *
  * @access   public
  * @param    string    foldername

  * @param    string    parentfolderid

  */
  public function doCreateFolder($foldername, $parentfolderid) {
    $ret = $this->cloudmail_office365_do_createfolder($foldername, $parentfolderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Deletes a message.
  *
  * @access   public
  * @param    string    id

  */
  public function doDelete($id) {
    $ret = $this->cloudmail_office365_do_delete($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Deletes an attachment.
  *
  * @access   public
  * @param    string    id

  * @param    string    attachmentid

  */
  public function doDeleteAttachment($id, $attachmentid) {
    $ret = $this->cloudmail_office365_do_deleteattachment($id, $attachmentid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Deletes a mail category.
  *
  * @access   public
  * @param    string    id

  */
  public function doDeleteCategory($id) {
    $ret = $this->cloudmail_office365_do_deletecategory($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Deletes a contact.
  *
  * @access   public
  * @param    string    id

  */
  public function doDeleteContact($id) {
    $ret = $this->cloudmail_office365_do_deletecontact($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Deletes a folder.
  *
  * @access   public
  * @param    string    id

  */
  public function doDeleteFolder($id) {
    $ret = $this->cloudmail_office365_do_deletefolder($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Forward a message.
  *
  * @access   public
  * @param    string    id

  * @param    string    messagehtml

  * @param    string    torecipients

  */
  public function doForward($id, $messagehtml, $torecipients) {
    $ret = $this->cloudmail_office365_do_forward($id, $messagehtml, $torecipients);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves a mail category.
  *
  * @access   public
  * @param    string    id

  */
  public function doGetCategory($id) {
    $ret = $this->cloudmail_office365_do_getcategory($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves the contact by Id.
  *
  * @access   public
  * @param    string    id

  */
  public function doGetContact($id) {
    $ret = $this->cloudmail_office365_do_getcontact($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves the contact property value by JSONPath.
  *
  * @access   public
  * @param    int    index

  * @param    string    jsonpath

  */
  public function doGetContactField($index, $jsonpath) {
    $ret = $this->cloudmail_office365_do_getcontactfield($index, $jsonpath);
    $err = $this->cloudmail_office365_get_last_error_code();
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves a folder.
  *
  * @access   public
  * @param    string    id

  */
  public function doGetFolder($id) {
    $ret = $this->cloudmail_office365_do_getfolder($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Interrupt the current method.
  *
  * @access   public
  */
  public function doInterrupt() {
    $ret = $this->cloudmail_office365_do_interrupt();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists all of a message's attachments.
  *
  * @access   public
  * @param    string    id

  */
  public function doListAttachments($id) {
    $ret = $this->cloudmail_office365_do_listattachments($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists all mail categories.
  *
  * @access   public
  */
  public function doListCategories() {
    $ret = $this->cloudmail_office365_do_listcategories();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists messages that have been changed within a specified folder.
  *
  * @access   public
  * @param    string    id

  * @param    int    maxpagesize

  */
  public function doListChanges($id, $maxpagesize) {
    $ret = $this->cloudmail_office365_do_listchanges($id, $maxpagesize);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists the contact folders found in the parent folder.
  *
  * @access   public
  * @param    string    parentfolderid

  */
  public function doListContactFolders($parentfolderid) {
    $ret = $this->cloudmail_office365_do_listcontactfolders($parentfolderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists the contacts in a folder.
  *
  * @access   public
  * @param    string    folderid

  */
  public function doListContacts($folderid) {
    $ret = $this->cloudmail_office365_do_listcontacts($folderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists the folders found in the parent folder.
  *
  * @access   public
  * @param    string    parentfolderid

  */
  public function doListFolders($parentfolderid) {
    $ret = $this->cloudmail_office365_do_listfolders($parentfolderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Lists the messages in a folder.
  *
  * @access   public
  * @param    string    folderid

  * @param    string    filter

  */
  public function doListMessages($folderid, $filter) {
    $ret = $this->cloudmail_office365_do_listmessages($folderid, $filter);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Moves a folder.
  *
  * @access   public
  * @param    string    id

  * @param    string    destfolderid

  */
  public function doMoveFolder($id, $destfolderid) {
    $ret = $this->cloudmail_office365_do_movefolder($id, $destfolderid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Moves a message.
  *
  * @access   public
  * @param    string    id

  * @param    string    destinationid

  */
  public function doMoveMessage($id, $destinationid) {
    $ret = $this->cloudmail_office365_do_movemessage($id, $destinationid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Renames a folder.
  *
  * @access   public
  * @param    string    folderid

  * @param    string    newname

  */
  public function doRenameFolder($folderid, $newname) {
    $ret = $this->cloudmail_office365_do_renamefolder($folderid, $newname);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Reply to a message.
  *
  * @access   public
  * @param    string    id

  * @param    string    messagehtml

  */
  public function doReply($id, $messagehtml) {
    $ret = $this->cloudmail_office365_do_reply($id, $messagehtml);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * ReplyAll to a message.
  *
  * @access   public
  * @param    string    id

  * @param    string    messagehtml

  */
  public function doReplyAll($id, $messagehtml) {
    $ret = $this->cloudmail_office365_do_replyall($id, $messagehtml);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * This method will reset the class.
  *
  * @access   public
  */
  public function doReset() {
    $ret = $this->cloudmail_office365_do_reset();
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves a message attachment.
  *
  * @access   public
  * @param    string    id

  * @param    string    attachmentid

  */
  public function doRetrieveAttachment($id, $attachmentid) {
    $ret = $this->cloudmail_office365_do_retrieveattachment($id, $attachmentid);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves a message.
  *
  * @access   public
  * @param    string    id

  */
  public function doRetrieveMessage($id) {
    $ret = $this->cloudmail_office365_do_retrievemessage($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Retrieves the raw message of the specified message ID.
  *
  * @access   public
  * @param    string    id

  */
  public function doRetrieveMessageRaw($id) {
    $ret = $this->cloudmail_office365_do_retrievemessageraw($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Search for messages.
  *
  * @access   public
  * @param    string    searchkey

  */
  public function doSearch($searchkey) {
    $ret = $this->cloudmail_office365_do_search($searchkey);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Send a custom HTTP request.
  *
  * @access   public
  * @param    string    httpmethod

  * @param    string    url

  * @param    string    postdata

  */
  public function doSendCustomRequest($httpmethod, $url, $postdata) {
    $ret = $this->cloudmail_office365_do_sendcustomrequest($httpmethod, $url, $postdata);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Sends an existing draft.
  *
  * @access   public
  * @param    string    id

  */
  public function doSendDraft($id) {
    $ret = $this->cloudmail_office365_do_senddraft($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Sends a new email.
  *
  * @access   public
  * @param    boolean    savetosentitems

  */
  public function doSendMail($savetosentitems) {
    $ret = $this->cloudmail_office365_do_sendmail($savetosentitems);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Sets the contact field value by JSONPath.
  *
  * @access   public
  * @param    int    index

  * @param    string    jsonpath

  * @param    string    value

  * @param    int    valuetype

  */
  public function doSetContactField($index, $jsonpath, $value, $valuetype) {
    $ret = $this->cloudmail_office365_do_setcontactfield($index, $jsonpath, $value, $valuetype);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Updates a message.
  *
  * @access   public
  * @param    string    id

  */
  public function doUpdate($id) {
    $ret = $this->cloudmail_office365_do_update($id);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Updates a category.
  *
  * @access   public
  * @param    string    id

  * @param    string    color

  */
  public function doUpdateCategory($id, $color) {
    $ret = $this->cloudmail_office365_do_updatecategory($id, $color);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


 /**
  * Updates a contact.
  *
  * @access   public
  * @param    int    index

  */
  public function doUpdateContact($index) {
    $ret = $this->cloudmail_office365_do_updatecontact($index);
    $err = $ret;
    if ($err != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }   


   

  public function VERSION() {
    return $this->cloudmail_office365_get(0);
  }

 /**
  * The number of records in the Attachment arrays.
  *
  * @access   public
  */
  public function getAttachmentCount() {
    $ret = $this->cloudmail_office365_get(1 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Attachment arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setAttachmentCount($value) {
    $ret = $this->cloudmail_office365_set(1, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the attachment type of the attachment.
  *
  * @access   public
  */
  public function getAttachmentAttachmentType($attachmentindex) {
    $ret = $this->cloudmail_office365_get(2 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the value of the unique content identifier of the attachment.
  *
  * @access   public
  */
  public function getAttachmentContentId($attachmentindex) {
    $ret = $this->cloudmail_office365_get(3 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the value of the unique content identifier of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setAttachmentContentId($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(3, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the content location of the attachment.
  *
  * @access   public
  */
  public function getAttachmentContentLocation($attachmentindex) {
    $ret = $this->cloudmail_office365_get(4 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the content type of the attachment.
  *
  * @access   public
  */
  public function getAttachmentContentType($attachmentindex) {
    $ret = $this->cloudmail_office365_get(5 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the raw data of the attachment.
  *
  * @access   public
  */
  public function getAttachmentData($attachmentindex) {
    $ret = $this->cloudmail_office365_get(6 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the raw data of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setAttachmentData($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(6, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the local file name associated with the attachment.
  *
  * @access   public
  */
  public function getAttachmentFileName($attachmentindex) {
    $ret = $this->cloudmail_office365_get(7 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the local file name associated with the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setAttachmentFileName($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(7, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the attachment identifier of the attachment.
  *
  * @access   public
  */
  public function getAttachmentId($attachmentindex) {
    $ret = $this->cloudmail_office365_get(8 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property is true if the attachment is an inline attachment.
  *
  * @access   public
  */
  public function getAttachmentIsInline($attachmentindex) {
    $ret = $this->cloudmail_office365_get(10 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property is true if the attachment is an inline attachment.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setAttachmentIsInline($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(10, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the date and time when the attachment was last modified.
  *
  * @access   public
  */
  public function getAttachmentLastModifiedDate($attachmentindex) {
    $ret = $this->cloudmail_office365_get(11 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the date and time when the attachment was last modified.
  *
  * @access   public
  * @param    string   value
  */
  public function setAttachmentLastModifiedDate($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(11, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the name of the attachment.
  *
  * @access   public
  */
  public function getAttachmentName($attachmentindex) {
    $ret = $this->cloudmail_office365_get(12 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the name of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setAttachmentName($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(12, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Provides the index of the parent item attachment when the RequestNestedItemAttachments configuration setting is enabled.
  *
  * @access   public
  */
  public function getAttachmentParentIndex($attachmentindex) {
    $ret = $this->cloudmail_office365_get(13 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the size in bytes of the attachment.
  *
  * @access   public
  */
  public function getAttachmentSize($attachmentindex) {
    $ret = $this->cloudmail_office365_get(14 , $attachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the size in bytes of the attachment.
  *
  * @access   public
  * @param    int   value
  */
  public function setAttachmentSize($attachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(14, $value , $attachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * An OAuth Authorization String.
  *
  * @access   public
  */
  public function getAuthorization() {
    $ret = $this->cloudmail_office365_get(15 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * An OAuth Authorization String.
  *
  * @access   public
  * @param    string   value
  */
  public function setAuthorization($value) {
    $ret = $this->cloudmail_office365_set(15, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the Category arrays.
  *
  * @access   public
  */
  public function getCategoryCount() {
    $ret = $this->cloudmail_office365_get(16 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Category arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setCategoryCount($value) {
    $ret = $this->cloudmail_office365_set(16, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the color of the category.
  *
  * @access   public
  */
  public function getCategoryColor($categoryindex) {
    $ret = $this->cloudmail_office365_get(17 , $categoryindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the display name of the category.
  *
  * @access   public
  */
  public function getCategoryDisplayName($categoryindex) {
    $ret = $this->cloudmail_office365_get(18 , $categoryindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the unique identifier of the category.
  *
  * @access   public
  */
  public function getCategoryId($categoryindex) {
    $ret = $this->cloudmail_office365_get(19 , $categoryindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The page marker for listing changed messages.
  *
  * @access   public
  */
  public function getChangeMarker() {
    $ret = $this->cloudmail_office365_get(20 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The page marker for listing changed messages.
  *
  * @access   public
  * @param    string   value
  */
  public function setChangeMarker($value) {
    $ret = $this->cloudmail_office365_set(20, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the ContactFolder arrays.
  *
  * @access   public
  */
  public function getContactFolderCount() {
    $ret = $this->cloudmail_office365_get(21 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the ContactFolder arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactFolderCount($value) {
    $ret = $this->cloudmail_office365_set(21, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier of the contact folder.
  *
  * @access   public
  */
  public function getContactFolderId($contactfolderindex) {
    $ret = $this->cloudmail_office365_get(22 , $contactfolderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The display name of the contact folder.
  *
  * @access   public
  */
  public function getContactFolderName($contactfolderindex) {
    $ret = $this->cloudmail_office365_get(23 , $contactfolderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier for the folder's parent.
  *
  * @access   public
  */
  public function getContactFolderParentFolderId($contactfolderindex) {
    $ret = $this->cloudmail_office365_get(24 , $contactfolderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the Contact arrays.
  *
  * @access   public
  */
  public function getContactsCount() {
    $ret = $this->cloudmail_office365_get(25 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Contact arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactsCount($value) {
    $ret = $this->cloudmail_office365_set(25, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the contact's company.
  *
  * @access   public
  */
  public function getContactCompanyName($contactindex) {
    $ret = $this->cloudmail_office365_get(26 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the contact's company.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactCompanyName($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(26, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The contact's display name.
  *
  * @access   public
  */
  public function getContactDisplayName($contactindex) {
    $ret = $this->cloudmail_office365_get(27 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The contact's display name.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactDisplayName($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(27, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The email address at the specified EmailAddressIndex.
  *
  * @access   public
  */
  public function getContactEmailAddress($contactindex) {
    $ret = $this->cloudmail_office365_get(28 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The email address at the specified EmailAddressIndex.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactEmailAddress($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(28, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * When retrieving contacts, it indicates the number of email addresses linked to the contact.
  *
  * @access   public
  */
  public function getContactEmailAddressesCount($contactindex) {
    $ret = $this->cloudmail_office365_get(29 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * When retrieving contacts, it indicates the number of email addresses linked to the contact.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactEmailAddressesCount($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(29, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Index of email address to get or set in the EmailAddress property.
  *
  * @access   public
  */
  public function getContactEmailAddressIndex($contactindex) {
    $ret = $this->cloudmail_office365_get(30 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Index of email address to get or set in the EmailAddress property.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactEmailAddressIndex($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(30, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The contact's first name.
  *
  * @access   public
  */
  public function getContactFirstName($contactindex) {
    $ret = $this->cloudmail_office365_get(31 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The contact's first name.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactFirstName($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(31, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier of the contact.
  *
  * @access   public
  */
  public function getContactId($contactindex) {
    $ret = $this->cloudmail_office365_get(32 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A JSON representation of the contact.
  *
  * @access   public
  */
  public function getContactJSON($contactindex) {
    $ret = $this->cloudmail_office365_get(33 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A JSON representation of the contact.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactJSON($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(33, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The contact's last name.
  *
  * @access   public
  */
  public function getContactLastName($contactindex) {
    $ret = $this->cloudmail_office365_get(34 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The contact's last name.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactLastName($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(34, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The user's notes about the contact.
  *
  * @access   public
  */
  public function getContactNotes($contactindex) {
    $ret = $this->cloudmail_office365_get(35 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The user's notes about the contact.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactNotes($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(35, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The ID of the contact's parent folder.
  *
  * @access   public
  */
  public function getContactParentId($contactindex) {
    $ret = $this->cloudmail_office365_get(36 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The ID of the contact's parent folder.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactParentId($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(36, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Index of phone number to get or set in the PhoneNumber property.
  *
  * @access   public
  */
  public function getContactPhoneIndex($contactindex) {
    $ret = $this->cloudmail_office365_get(37 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Index of phone number to get or set in the PhoneNumber property.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactPhoneIndex($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(37, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property is used to get or set a phone number at the index specified by PhoneIndex.
  *
  * @access   public
  */
  public function getContactPhoneNumber($contactindex) {
    $ret = $this->cloudmail_office365_get(38 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property is used to get or set a phone number at the index specified by PhoneIndex.
  *
  * @access   public
  * @param    string   value
  */
  public function setContactPhoneNumber($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(38, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Number of the contact's phone numbers.
  *
  * @access   public
  */
  public function getContactPhonesCount($contactindex) {
    $ret = $this->cloudmail_office365_get(39 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Number of the contact's phone numbers.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactPhonesCount($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(39, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The phone number type.
  *
  * @access   public
  */
  public function getContactPhoneType($contactindex) {
    $ret = $this->cloudmail_office365_get(40 , $contactindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The phone number type.
  *
  * @access   public
  * @param    int   value
  */
  public function setContactPhoneType($contactindex, $value) {
    $ret = $this->cloudmail_office365_set(40, $value , $contactindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether to automatically detect and use firewall system settings, if available.
  *
  * @access   public
  */
  public function getFirewallAutoDetect() {
    $ret = $this->cloudmail_office365_get(41 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether to automatically detect and use firewall system settings, if available.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setFirewallAutoDetect($value) {
    $ret = $this->cloudmail_office365_set(41, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of firewall to connect through.
  *
  * @access   public
  */
  public function getFirewallType() {
    $ret = $this->cloudmail_office365_get(42 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of firewall to connect through.
  *
  * @access   public
  * @param    int   value
  */
  public function setFirewallType($value) {
    $ret = $this->cloudmail_office365_set(42, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name or IP address of the firewall (optional).
  *
  * @access   public
  */
  public function getFirewallHost() {
    $ret = $this->cloudmail_office365_get(43 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The name or IP address of the firewall (optional).
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallHost($value) {
    $ret = $this->cloudmail_office365_set(43, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A password if authentication is to be used when connecting through the firewall.
  *
  * @access   public
  */
  public function getFirewallPassword() {
    $ret = $this->cloudmail_office365_get(44 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A password if authentication is to be used when connecting through the firewall.
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallPassword($value) {
    $ret = $this->cloudmail_office365_set(44, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The Transmission Control Protocol (TCP) port for the firewall Host .
  *
  * @access   public
  */
  public function getFirewallPort() {
    $ret = $this->cloudmail_office365_get(45 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The Transmission Control Protocol (TCP) port for the firewall Host .
  *
  * @access   public
  * @param    int   value
  */
  public function setFirewallPort($value) {
    $ret = $this->cloudmail_office365_set(45, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A username if authentication is to be used when connecting through a firewall.
  *
  * @access   public
  */
  public function getFirewallUser() {
    $ret = $this->cloudmail_office365_get(46 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A username if authentication is to be used when connecting through a firewall.
  *
  * @access   public
  * @param    string   value
  */
  public function setFirewallUser($value) {
    $ret = $this->cloudmail_office365_set(46, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the Folder arrays.
  *
  * @access   public
  */
  public function getFolderCount() {
    $ret = $this->cloudmail_office365_get(47 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the Folder arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setFolderCount($value) {
    $ret = $this->cloudmail_office365_set(47, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of child folders the folder has.
  *
  * @access   public
  */
  public function getFolderChildFolderCount($folderindex) {
    $ret = $this->cloudmail_office365_get(48 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The child folders of the folder.
  *
  * @access   public
  */
  public function getFolderChildFolders($folderindex) {
    $ret = $this->cloudmail_office365_get(49 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The display name of the folder.
  *
  * @access   public
  */
  public function getFolderDisplayName($folderindex) {
    $ret = $this->cloudmail_office365_get(50 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier of the folder.
  *
  * @access   public
  */
  public function getFolderId($folderindex) {
    $ret = $this->cloudmail_office365_get(51 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The message rules of the folder.
  *
  * @access   public
  */
  public function getFolderMessageRules($folderindex) {
    $ret = $this->cloudmail_office365_get(52 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The messages contained in the folder.
  *
  * @access   public
  */
  public function getFolderMessages($folderindex) {
    $ret = $this->cloudmail_office365_get(53 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The multi-value extended properties defined for the folder.
  *
  * @access   public
  */
  public function getFolderMultiValueExtendedProperties($folderindex) {
    $ret = $this->cloudmail_office365_get(54 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier for the folder's parent.
  *
  * @access   public
  */
  public function getFolderParentFolderId($folderindex) {
    $ret = $this->cloudmail_office365_get(55 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The single-value extended properties defined for the folder.
  *
  * @access   public
  */
  public function getFolderSingleValueExtendedProperties($folderindex) {
    $ret = $this->cloudmail_office365_get(56 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of total items in the folder.
  *
  * @access   public
  */
  public function getFolderTotalItemCount($folderindex) {
    $ret = $this->cloudmail_office365_get(57 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of unread items in the folder.
  *
  * @access   public
  */
  public function getFolderUnreadItemCount($folderindex) {
    $ret = $this->cloudmail_office365_get(58 , $folderindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The page marker for listing contacts.
  *
  * @access   public
  */
  public function getListContactsMarker() {
    $ret = $this->cloudmail_office365_get(59 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The page marker for listing contacts.
  *
  * @access   public
  * @param    string   value
  */
  public function setListContactsMarker($value) {
    $ret = $this->cloudmail_office365_set(59, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The page marker for listing folders.
  *
  * @access   public
  */
  public function getListFoldersMarker() {
    $ret = $this->cloudmail_office365_get(60 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The page marker for listing folders.
  *
  * @access   public
  * @param    string   value
  */
  public function setListFoldersMarker($value) {
    $ret = $this->cloudmail_office365_set(60, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The page marker for listing messages.
  *
  * @access   public
  */
  public function getListMessagesMarker() {
    $ret = $this->cloudmail_office365_get(61 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The page marker for listing messages.
  *
  * @access   public
  * @param    string   value
  */
  public function setListMessagesMarker($value) {
    $ret = $this->cloudmail_office365_set(61, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Provides the raw message content.
  *
  * @access   public
  */
  public function getMessage() {
    $ret = $this->cloudmail_office365_get(62 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the MessageAttachment arrays.
  *
  * @access   public
  */
  public function getMessageAttachmentCount() {
    $ret = $this->cloudmail_office365_get(63 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the MessageAttachment arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setMessageAttachmentCount($value) {
    $ret = $this->cloudmail_office365_set(63, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the attachment type of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentAttachmentType($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(64 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the value of the unique content identifier of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentContentId($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(65 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the value of the unique content identifier of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageAttachmentContentId($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(65, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the content location of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentContentLocation($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(66 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the content type of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentContentType($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(67 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the raw data of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentData($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(68 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the raw data of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageAttachmentData($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(68, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the local file name associated with the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentFileName($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(69 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the local file name associated with the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageAttachmentFileName($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(69, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the attachment identifier of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentId($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(70 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property is true if the attachment is an inline attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentIsInline($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(72 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property is true if the attachment is an inline attachment.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageAttachmentIsInline($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(72, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the date and time when the attachment was last modified.
  *
  * @access   public
  */
  public function getMessageAttachmentLastModifiedDate($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(73 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the date and time when the attachment was last modified.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageAttachmentLastModifiedDate($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(73, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the name of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentName($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(74 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the name of the attachment.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageAttachmentName($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(74, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Provides the index of the parent item attachment when the RequestNestedItemAttachments configuration setting is enabled.
  *
  * @access   public
  */
  public function getMessageAttachmentParentIndex($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(75 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * This property contains the size in bytes of the attachment.
  *
  * @access   public
  */
  public function getMessageAttachmentSize($messageattachmentindex) {
    $ret = $this->cloudmail_office365_get(76 , $messageattachmentindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * This property contains the size in bytes of the attachment.
  *
  * @access   public
  * @param    int   value
  */
  public function setMessageAttachmentSize($messageattachmentindex, $value) {
    $ret = $this->cloudmail_office365_set(76, $value , $messageattachmentindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma separated list of recipients for blind carbon copies for a message.
  *
  * @access   public
  */
  public function getMessageBcc() {
    $ret = $this->cloudmail_office365_get(77 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A comma separated list of recipients for blind carbon copies for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageBcc($value) {
    $ret = $this->cloudmail_office365_set(77, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The body content for a message.
  *
  * @access   public
  */
  public function getMessageBodyContent() {
    $ret = $this->cloudmail_office365_get(78 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The body content for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageBodyContent($value) {
    $ret = $this->cloudmail_office365_set(78, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The body content type for a message.
  *
  * @access   public
  */
  public function getMessageBodyContentType() {
    $ret = $this->cloudmail_office365_get(79 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The body content type for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageBodyContentType($value) {
    $ret = $this->cloudmail_office365_set(79, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma separated list of recipients for carbon copies for a message.
  *
  * @access   public
  */
  public function getMessageCc() {
    $ret = $this->cloudmail_office365_get(80 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A comma separated list of recipients for carbon copies for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageCc($value) {
    $ret = $this->cloudmail_office365_set(80, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a message will request a Delivery Receipt.
  *
  * @access   public
  */
  public function getMessageDeliveryReceipt() {
    $ret = $this->cloudmail_office365_get(81 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether or not a message will request a Delivery Receipt.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageDeliveryReceipt($value) {
    $ret = $this->cloudmail_office365_set(81, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The author of a message.
  *
  * @access   public
  */
  public function getMessageFrom() {
    $ret = $this->cloudmail_office365_get(82 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The author of a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageFrom($value) {
    $ret = $this->cloudmail_office365_set(82, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The importance of a message.
  *
  * @access   public
  */
  public function getMessageImportance() {
    $ret = $this->cloudmail_office365_get(83 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The importance of a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageImportance($value) {
    $ret = $this->cloudmail_office365_set(83, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The number of records in the MessageInfo arrays.
  *
  * @access   public
  */
  public function getMessageInfoCount() {
    $ret = $this->cloudmail_office365_get(84 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The number of records in the MessageInfo arrays.
  *
  * @access   public
  * @param    int   value
  */
  public function setMessageInfoCount($value) {
    $ret = $this->cloudmail_office365_set(84, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The BCc recipients of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoBcc($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(85 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The BCc recipients of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoBcc($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(85, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The body content of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoBodyContent($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(86 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The body content of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoBodyContent($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(86, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The body content type (e.
  *
  * @access   public
  */
  public function getMessageInfoBodyContentType($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(87 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The body content type (e.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoBodyContentType($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(87, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The body preview of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoBodyPreview($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(88 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The categories of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoCategories($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(89 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The categories of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoCategories($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(89, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The Cc recipients of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoCc($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(90 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The Cc recipients of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoCc($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(90, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The conversation unique identifier of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoConversationId($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(91 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The conversation index of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoConversationIndex($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(92 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date created of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoCreatedDate($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(93 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a delivery receipt was requested for a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoDeliveryReceiptRequested($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(94 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether or not a delivery receipt was requested for a message in a message info listing.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageInfoDeliveryReceiptRequested($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(94, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Message flag in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoFlagStatus($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(95 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Message flag in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoFlagStatus($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(95, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The sender of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoFrom($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(96 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a message in a message info listing has attachments.
  *
  * @access   public
  */
  public function getMessageInfoHasAttachments($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(97 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier of a message in a message info listing set by Microsoft.
  *
  * @access   public
  */
  public function getMessageInfoId($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(98 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The importance of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoImportance($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(99 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The importance of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoImportance($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(99, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The inference classification of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoInferenceClassification($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(100 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a message in a message info listing is a draft.
  *
  * @access   public
  */
  public function getMessageInfoIsDraft($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(101 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a message in a message info listing has been read.
  *
  * @access   public
  */
  public function getMessageInfoIsRead($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(102 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether or not a message in a message info listing has been read.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageInfoIsRead($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(102, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The full JSON content of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoJSON($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(103 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The full JSON content of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoJSON($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(103, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The last modified date of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoLastModifiedDate($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(104 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The message headers of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoMessageHeaders($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(105 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The internet message id for the message as described by rfc2822.
  *
  * @access   public
  */
  public function getMessageInfoMessageId($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(106 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The unique identifier of the parent folder of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoParentFolderId($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(107 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a read receipt was requested for a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoReadReceiptRequested($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(108 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether or not a read receipt was requested for a message in a message info listing.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageInfoReadReceiptRequested($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(108, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The received date of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoReceivedDate($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(109 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Where to send replies for a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoReplyTo($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(110 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Where to send replies for a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoReplyTo($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(110, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The sender of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoSender($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(111 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date a message was sent for a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoSentDate($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(112 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoSubject($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(113 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoSubject($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(113, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The recipients of a message in a message info listing.
  *
  * @access   public
  */
  public function getMessageInfoTo($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(114 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The recipients of a message in a message info listing.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageInfoTo($messageinfoindex, $value) {
    $ret = $this->cloudmail_office365_set(114, $value , $messageinfoindex);
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL to open a message in a message info listing in Outlook on the web.
  *
  * @access   public
  */
  public function getMessageInfoWebLink($messageinfoindex) {
    $ret = $this->cloudmail_office365_get(115 , $messageinfoindex);
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The additional message headers for a message.
  *
  * @access   public
  */
  public function getMessageOtherHeaders() {
    $ret = $this->cloudmail_office365_get(116 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The additional message headers for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageOtherHeaders($value) {
    $ret = $this->cloudmail_office365_set(116, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether or not a message will request a Read Receipt.
  *
  * @access   public
  */
  public function getMessageReadReceipt() {
    $ret = $this->cloudmail_office365_get(117 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether or not a message will request a Read Receipt.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setMessageReadReceipt($value) {
    $ret = $this->cloudmail_office365_set(117, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A mail address to reply to.
  *
  * @access   public
  */
  public function getMessageReplyTo() {
    $ret = $this->cloudmail_office365_get(118 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A mail address to reply to.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageReplyTo($value) {
    $ret = $this->cloudmail_office365_set(118, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of a message.
  *
  * @access   public
  */
  public function getMessageSubject() {
    $ret = $this->cloudmail_office365_get(119 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageSubject($value) {
    $ret = $this->cloudmail_office365_set(119, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma separated list of recipients for a message.
  *
  * @access   public
  */
  public function getMessageTo() {
    $ret = $this->cloudmail_office365_get(120 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A comma separated list of recipients for a message.
  *
  * @access   public
  * @param    string   value
  */
  public function setMessageTo($value) {
    $ret = $this->cloudmail_office365_set(120, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A marker indicating which page of changes to return in the future.
  *
  * @access   public
  */
  public function getNextChangeMarker() {
    $ret = $this->cloudmail_office365_get(121 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A marker indicating which page of changes to return in the future.
  *
  * @access   public
  * @param    string   value
  */
  public function setNextChangeMarker($value) {
    $ret = $this->cloudmail_office365_set(121, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The access token returned by the authorization server.
  *
  * @access   public
  */
  public function getOAuthAccessToken() {
    $ret = $this->cloudmail_office365_get(122 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The access token returned by the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthAccessToken($value) {
    $ret = $this->cloudmail_office365_set(122, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The authorization code that is exchanged for an access token.
  *
  * @access   public
  */
  public function getOAuthAuthorizationCode() {
    $ret = $this->cloudmail_office365_get(123 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The authorization code that is exchanged for an access token.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthAuthorizationCode($value) {
    $ret = $this->cloudmail_office365_set(123, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The scope request or response parameter used during authorization.
  *
  * @access   public
  */
  public function getOAuthAuthorizationScope() {
    $ret = $this->cloudmail_office365_get(124 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The scope request or response parameter used during authorization.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthAuthorizationScope($value) {
    $ret = $this->cloudmail_office365_set(124, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The id of the client assigned when registering the application.
  *
  * @access   public
  */
  public function getOAuthClientId() {
    $ret = $this->cloudmail_office365_get(125 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The id of the client assigned when registering the application.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthClientId($value) {
    $ret = $this->cloudmail_office365_set(125, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of client that is requesting authorization.
  *
  * @access   public
  */
  public function getOAuthClientProfile() {
    $ret = $this->cloudmail_office365_get(126 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of client that is requesting authorization.
  *
  * @access   public
  * @param    int   value
  */
  public function setOAuthClientProfile($value) {
    $ret = $this->cloudmail_office365_set(126, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The secret value for the client assigned when registering the application.
  *
  * @access   public
  */
  public function getOAuthClientSecret() {
    $ret = $this->cloudmail_office365_get(127 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The secret value for the client assigned when registering the application.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthClientSecret($value) {
    $ret = $this->cloudmail_office365_set(127, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The OAuth grant type used to acquire an OAuth access token.
  *
  * @access   public
  */
  public function getOAuthGrantType() {
    $ret = $this->cloudmail_office365_get(128 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The OAuth grant type used to acquire an OAuth access token.
  *
  * @access   public
  * @param    int   value
  */
  public function setOAuthGrantType($value) {
    $ret = $this->cloudmail_office365_set(128, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Specifies the refresh token received from or sent to the authorization server.
  *
  * @access   public
  */
  public function getOAuthRefreshToken() {
    $ret = $this->cloudmail_office365_get(129 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Specifies the refresh token received from or sent to the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthRefreshToken($value) {
    $ret = $this->cloudmail_office365_set(129, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Specifies whether the class will request a refresh token during authorization.
  *
  * @access   public
  */
  public function getOAuthRequestRefreshToken() {
    $ret = $this->cloudmail_office365_get(130 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Specifies whether the class will request a refresh token during authorization.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setOAuthRequestRefreshToken($value) {
    $ret = $this->cloudmail_office365_set(130, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL where the user (browser) returns after authenticating.
  *
  * @access   public
  */
  public function getOAuthReturnURL() {
    $ret = $this->cloudmail_office365_get(131 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL where the user (browser) returns after authenticating.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthReturnURL($value) {
    $ret = $this->cloudmail_office365_set(131, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL of the authorization server.
  *
  * @access   public
  */
  public function getOAuthServerAuthURL() {
    $ret = $this->cloudmail_office365_get(132 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL of the authorization server.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthServerAuthURL($value) {
    $ret = $this->cloudmail_office365_set(132, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL of the token server used to obtain the access token.
  *
  * @access   public
  */
  public function getOAuthServerTokenURL() {
    $ret = $this->cloudmail_office365_get(133 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The URL of the token server used to obtain the access token.
  *
  * @access   public
  * @param    string   value
  */
  public function setOAuthServerTokenURL($value) {
    $ret = $this->cloudmail_office365_set(133, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The URL to which the user should be re-directed for authorization.
  *
  * @access   public
  */
  public function getOAuthWebAuthURL() {
    $ret = $this->cloudmail_office365_get(134 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of authorization to perform when connecting to the proxy.
  *
  * @access   public
  */
  public function getProxyAuthScheme() {
    $ret = $this->cloudmail_office365_get(135 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of authorization to perform when connecting to the proxy.
  *
  * @access   public
  * @param    int   value
  */
  public function setProxyAuthScheme($value) {
    $ret = $this->cloudmail_office365_set(135, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether to automatically detect and use proxy system settings, if available.
  *
  * @access   public
  */
  public function getProxyAutoDetect() {
    $ret = $this->cloudmail_office365_get(136 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * Whether to automatically detect and use proxy system settings, if available.
  *
  * @access   public
  * @param    boolean   value
  */
  public function setProxyAutoDetect($value) {
    $ret = $this->cloudmail_office365_set(136, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A password if authentication is to be used for the proxy.
  *
  * @access   public
  */
  public function getProxyPassword() {
    $ret = $this->cloudmail_office365_get(137 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A password if authentication is to be used for the proxy.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyPassword($value) {
    $ret = $this->cloudmail_office365_set(137, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The Transmission Control Protocol (TCP) port for the proxy Server (default 80).
  *
  * @access   public
  */
  public function getProxyPort() {
    $ret = $this->cloudmail_office365_get(138 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The Transmission Control Protocol (TCP) port for the proxy Server (default 80).
  *
  * @access   public
  * @param    int   value
  */
  public function setProxyPort($value) {
    $ret = $this->cloudmail_office365_set(138, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * If a proxy Server is given, then the HTTP request is sent to the proxy instead of the server otherwise specified.
  *
  * @access   public
  */
  public function getProxyServer() {
    $ret = $this->cloudmail_office365_get(139 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * If a proxy Server is given, then the HTTP request is sent to the proxy instead of the server otherwise specified.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyServer($value) {
    $ret = $this->cloudmail_office365_set(139, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * When to use a Secure Sockets Layer (SSL) for the connection to the proxy.
  *
  * @access   public
  */
  public function getProxySSL() {
    $ret = $this->cloudmail_office365_get(140 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * When to use a Secure Sockets Layer (SSL) for the connection to the proxy.
  *
  * @access   public
  * @param    int   value
  */
  public function setProxySSL($value) {
    $ret = $this->cloudmail_office365_set(140, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A username if authentication is to be used for the proxy.
  *
  * @access   public
  */
  public function getProxyUser() {
    $ret = $this->cloudmail_office365_get(141 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * A username if authentication is to be used for the proxy.
  *
  * @access   public
  * @param    string   value
  */
  public function setProxyUser($value) {
    $ret = $this->cloudmail_office365_set(141, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The parts of a message that should be retrieved.
  *
  * @access   public
  */
  public function getSelect() {
    $ret = $this->cloudmail_office365_get(142 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The parts of a message that should be retrieved.
  *
  * @access   public
  * @param    string   value
  */
  public function setSelect($value) {
    $ret = $this->cloudmail_office365_set(142, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertEffectiveDate() {
    $ret = $this->cloudmail_office365_get(143 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertExpirationDate() {
    $ret = $this->cloudmail_office365_get(144 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertExtendedKeyUsage() {
    $ret = $this->cloudmail_office365_get(145 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprint() {
    $ret = $this->cloudmail_office365_get(146 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprintSHA1() {
    $ret = $this->cloudmail_office365_get(147 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertFingerprintSHA256() {
    $ret = $this->cloudmail_office365_get(148 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertIssuer() {
    $ret = $this->cloudmail_office365_get(149 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKey() {
    $ret = $this->cloudmail_office365_get(150 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_office365_get(151 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPrivateKeyContainer() {
    $ret = $this->cloudmail_office365_get(152 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKey() {
    $ret = $this->cloudmail_office365_get(153 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_office365_get(154 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertPublicKeyLength() {
    $ret = $this->cloudmail_office365_get(155 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSerialNumber() {
    $ret = $this->cloudmail_office365_get(156 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSignatureAlgorithm() {
    $ret = $this->cloudmail_office365_get(157 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStore() {
    $ret = $this->cloudmail_office365_get(158 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertStore($value) {
    $ret = $this->cloudmail_office365_set(158, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStorePassword() {
    $ret = $this->cloudmail_office365_get(159 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertStorePassword($value) {
    $ret = $this->cloudmail_office365_set(159, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertStoreType() {
    $ret = $this->cloudmail_office365_get(160 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLAcceptServerCertStoreType($value) {
    $ret = $this->cloudmail_office365_set(160, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSubjectAltNames() {
    $ret = $this->cloudmail_office365_get(161 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintMD5() {
    $ret = $this->cloudmail_office365_get(162 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintSHA1() {
    $ret = $this->cloudmail_office365_get(163 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertThumbprintSHA256() {
    $ret = $this->cloudmail_office365_get(164 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLAcceptServerCertUsage() {
    $ret = $this->cloudmail_office365_get(165 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertUsageFlags() {
    $ret = $this->cloudmail_office365_get(166 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertVersion() {
    $ret = $this->cloudmail_office365_get(167 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLAcceptServerCertSubject() {
    $ret = $this->cloudmail_office365_get(168 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertSubject($value) {
    $ret = $this->cloudmail_office365_set(168, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLAcceptServerCertEncoded() {
    $ret = $this->cloudmail_office365_get(169 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLAcceptServerCertEncoded($value) {
    $ret = $this->cloudmail_office365_set(169, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLCertEffectiveDate() {
    $ret = $this->cloudmail_office365_get(170 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLCertExpirationDate() {
    $ret = $this->cloudmail_office365_get(171 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLCertExtendedKeyUsage() {
    $ret = $this->cloudmail_office365_get(172 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprint() {
    $ret = $this->cloudmail_office365_get(173 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprintSHA1() {
    $ret = $this->cloudmail_office365_get(174 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLCertFingerprintSHA256() {
    $ret = $this->cloudmail_office365_get(175 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLCertIssuer() {
    $ret = $this->cloudmail_office365_get(176 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLCertPrivateKey() {
    $ret = $this->cloudmail_office365_get(177 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_office365_get(178 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLCertPrivateKeyContainer() {
    $ret = $this->cloudmail_office365_get(179 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLCertPublicKey() {
    $ret = $this->cloudmail_office365_get(180 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_office365_get(181 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLCertPublicKeyLength() {
    $ret = $this->cloudmail_office365_get(182 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLCertSerialNumber() {
    $ret = $this->cloudmail_office365_get(183 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLCertSignatureAlgorithm() {
    $ret = $this->cloudmail_office365_get(184 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLCertStore() {
    $ret = $this->cloudmail_office365_get(185 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertStore($value) {
    $ret = $this->cloudmail_office365_set(185, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLCertStorePassword() {
    $ret = $this->cloudmail_office365_get(186 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertStorePassword($value) {
    $ret = $this->cloudmail_office365_set(186, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLCertStoreType() {
    $ret = $this->cloudmail_office365_get(187 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLCertStoreType($value) {
    $ret = $this->cloudmail_office365_set(187, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLCertSubjectAltNames() {
    $ret = $this->cloudmail_office365_get(188 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintMD5() {
    $ret = $this->cloudmail_office365_get(189 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintSHA1() {
    $ret = $this->cloudmail_office365_get(190 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLCertThumbprintSHA256() {
    $ret = $this->cloudmail_office365_get(191 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLCertUsage() {
    $ret = $this->cloudmail_office365_get(192 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLCertUsageFlags() {
    $ret = $this->cloudmail_office365_get(193 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLCertVersion() {
    $ret = $this->cloudmail_office365_get(194 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLCertSubject() {
    $ret = $this->cloudmail_office365_get(195 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertSubject($value) {
    $ret = $this->cloudmail_office365_set(195, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLCertEncoded() {
    $ret = $this->cloudmail_office365_get(196 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  * @param    string   value
  */
  public function setSSLCertEncoded($value) {
    $ret = $this->cloudmail_office365_set(196, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The Secure Sockets Layer/Transport Layer Security (SSL/TLS) implementation to use.
  *
  * @access   public
  */
  public function getSSLProvider() {
    $ret = $this->cloudmail_office365_get(197 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
 /**
  * The Secure Sockets Layer/Transport Layer Security (SSL/TLS) implementation to use.
  *
  * @access   public
  * @param    int   value
  */
  public function setSSLProvider($value) {
    $ret = $this->cloudmail_office365_set(197, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which this certificate becomes valid.
  *
  * @access   public
  */
  public function getSSLServerCertEffectiveDate() {
    $ret = $this->cloudmail_office365_get(198 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The date on which the certificate expires.
  *
  * @access   public
  */
  public function getSSLServerCertExpirationDate() {
    $ret = $this->cloudmail_office365_get(199 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * A comma-delimited list of extended key usage identifiers.
  *
  * @access   public
  */
  public function getSSLServerCertExtendedKeyUsage() {
    $ret = $this->cloudmail_office365_get(200 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 16-byte MD5 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprint() {
    $ret = $this->cloudmail_office365_get(201 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 20-byte SHA-1 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprintSHA1() {
    $ret = $this->cloudmail_office365_get(202 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The hex-encoded, 32-byte SHA-256 fingerprint of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertFingerprintSHA256() {
    $ret = $this->cloudmail_office365_get(203 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The issuer of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertIssuer() {
    $ret = $this->cloudmail_office365_get(204 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The private key of the certificate (if available).
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKey() {
    $ret = $this->cloudmail_office365_get(205 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Whether a PrivateKey is available for the  selected certificate.
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKeyAvailable() {
    $ret = $this->cloudmail_office365_get(206 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the PrivateKey container for the  certificate (if available).
  *
  * @access   public
  */
  public function getSSLServerCertPrivateKeyContainer() {
    $ret = $this->cloudmail_office365_get(207 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The public key of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertPublicKey() {
    $ret = $this->cloudmail_office365_get(208 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The textual description of the certificate's public key algorithm.
  *
  * @access   public
  */
  public function getSSLServerCertPublicKeyAlgorithm() {
    $ret = $this->cloudmail_office365_get(209 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The length of the certificate's public key (in bits).
  *
  * @access   public
  */
  public function getSSLServerCertPublicKeyLength() {
    $ret = $this->cloudmail_office365_get(210 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The serial number of the certificate encoded as a  string.
  *
  * @access   public
  */
  public function getSSLServerCertSerialNumber() {
    $ret = $this->cloudmail_office365_get(211 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of the certificate's signature algorithm.
  *
  * @access   public
  */
  public function getSSLServerCertSignatureAlgorithm() {
    $ret = $this->cloudmail_office365_get(212 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The name of the certificate store for the client certificate.
  *
  * @access   public
  */
  public function getSSLServerCertStore() {
    $ret = $this->cloudmail_office365_get(213 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * If the type of certificate store requires  a password, this property is used to specify the password needed to open the certificate store.
  *
  * @access   public
  */
  public function getSSLServerCertStorePassword() {
    $ret = $this->cloudmail_office365_get(214 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The type of certificate store for this certificate.
  *
  * @access   public
  */
  public function getSSLServerCertStoreType() {
    $ret = $this->cloudmail_office365_get(215 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * Comma-separated lists of alternative subject names for the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertSubjectAltNames() {
    $ret = $this->cloudmail_office365_get(216 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The MD5 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintMD5() {
    $ret = $this->cloudmail_office365_get(217 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-1 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintSHA1() {
    $ret = $this->cloudmail_office365_get(218 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The SHA-256 hash of the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertThumbprintSHA256() {
    $ret = $this->cloudmail_office365_get(219 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The text description of UsageFlags .
  *
  * @access   public
  */
  public function getSSLServerCertUsage() {
    $ret = $this->cloudmail_office365_get(220 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The flags that show intended use for the certificate.
  *
  * @access   public
  */
  public function getSSLServerCertUsageFlags() {
    $ret = $this->cloudmail_office365_get(221 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate's version number.
  *
  * @access   public
  */
  public function getSSLServerCertVersion() {
    $ret = $this->cloudmail_office365_get(222 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The subject of the certificate used for client authentication.
  *
  * @access   public
  */
  public function getSSLServerCertSubject() {
    $ret = $this->cloudmail_office365_get(223 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }


 /**
  * The certificate (PEM/Base64 encoded).
  *
  * @access   public
  */
  public function getSSLServerCertEncoded() {
    $ret = $this->cloudmail_office365_get(224 );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }



  public function getRuntimeLicense() {
    $ret = $this->cloudmail_office365_get(8001/*PID_RUNTIME_LICENSE*/ );
    $error_code = $this->cloudmail_office365_get_last_error_code();
    if ($error_code != 0) {
      throw new Exception($error_code . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }

  public function setRuntimeLicense($value) {
    $ret = $this->cloudmail_office365_set(8001/*PID_RUNTIME_LICENSE*/, $value );
    if ($ret != 0) {
      throw new Exception($ret . ": " . $this->cloudmail_office365_get_last_error());
    }
    return $ret;
  }
  
 /**
  * Fired when an attachment is retrieved from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, name, contenttype, contentlocation, contentbytes, contentid, lastmodifieddatetime, isinline, size    
  */
  public function fireAttachmentList($param) {
    return $param;
  }


 /**
  * Fired when an attachment is retrieved from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, displayname, color    
  */
  public function fireCategoryList($param) {
    return $param;
  }


 /**
  * Fired when a contact folder is retrieved by the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, name, parentfolderid    
  */
  public function fireContactFolderList($param) {
    return $param;
  }


 /**
  * Fired when a contact is retrieved from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, parentid, displayname, firstemail    
  */
  public function fireContactList($param) {
    return $param;
  }


 /**
  * Fired when information is available about errors during data delivery.
  *
  * @access   public
  * @param    array   Array of event parameters: errorcode, description    
  */
  public function fireError($param) {
    return $param;
  }


 /**
  * Fired when a folder is retrieved by the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, displayname, parentfolderid, childfoldercount, unreaditemcount, totalitemcount    
  */
  public function fireFolderList($param) {
    return $param;
  }


 /**
  * Fired once for each log message.
  *
  * @access   public
  * @param    array   Array of event parameters: loglevel, message, logtype    
  */
  public function fireLog($param) {
    return $param;
  }


 /**
  * Fired when a message is retrieved from the server.
  *
  * @access   public
  * @param    array   Array of event parameters: id, subject, messagesender, receiveddatetime, sentdatetime, hasattachments, bodycontenttype, bodycontent, bodypreview    
  */
  public function fireMessageList($param) {
    return $param;
  }


 /**
  * Fired after the server presents its certificate to the client.
  *
  * @access   public
  * @param    array   Array of event parameters: certencoded, certsubject, certissuer, status, accept    
  */
  public function fireSSLServerAuthentication($param) {
    return $param;
  }


 /**
  * Fired when secure connection progress messages are available.
  *
  * @access   public
  * @param    array   Array of event parameters: message    
  */
  public function fireSSLStatus($param) {
    return $param;
  }


 /**
  * Fired while a document transfers (delivers document).
  *
  * @access   public
  * @param    array   Array of event parameters: direction, bytestransferred, percentdone, text    
  */
  public function fireTransfer($param) {
    return $param;
  }



  private function cloudmail_office365_open($key) {
    global $lib_cloudmail_office365;
    global $objMap;

    $id = spl_object_id($this);
    $objMap[$id] = $this;
    $objId = $lib_cloudmail_office365->cast("void*", $id);

    $this->handle = $lib_cloudmail_office365->CloudMail_Office365_Create("cloudmail_office365_callback", $objId, null, 0);
    if (!isset($this->handle)) {
      error_log("out of memory", 0);
    }

    $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 8004/*PID_KEYCHECK_PHP*/, 0, $key, 0);
  }

  private function cloudmail_office365_close() {
    global $lib_cloudmail_office365;
    global $objMap;

    if (isset($this->handle)) {
      $lib_cloudmail_office365->CloudMail_Office365_Destroy($this->handle);
      $this->handle = null;
    }
    
    $objMap[spl_object_id($this)] = null;
  }

  private function cloudmail_office365_get_last_error() {
    global $lib_cloudmail_office365;
    $retval = $lib_cloudmail_office365->CloudMail_Office365_GetLastError($this->handle);
    if (isset($retval)) return FFI::string($lib_cloudmail_office365->cast("char*", $retval));
  }

  private function cloudmail_office365_get_last_error_code() {
    global $lib_cloudmail_office365;
    return (int)$lib_cloudmail_office365->CloudMail_Office365_GetLastErrorCode($this->handle);
  }
  

 /**
  * Adds a file attachment to an existing message.
  *
  * @access   private
  * @param    string    messageid

  * @param    string    name

  * @param    string    localfile

  */
  private function cloudmail_office365_do_addattachment($messageid, $name, $localfile) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 3 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 3 + 1 . "]", false, true);
    $cstr = $messageid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $name . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $localfile . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $param[3] = null;
    $cbparam[3] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 2, 3, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Adds an item attachment to an existing message.
  *
  * @access   private
  * @param    string    messageid

  * @param    string    msgtoattachid

  * @param    string    itemjson

  */
  private function cloudmail_office365_do_additemattachment($messageid, $msgtoattachid, $itemjson) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 3 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 3 + 1 . "]", false, true);
    $cstr = $messageid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $msgtoattachid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $itemjson . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $param[3] = null;
    $cbparam[3] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 3, 3, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Get the authorization string required to access the protected resource.
  *
  * @access   private
  */
  private function cloudmail_office365_do_authorize() {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 4, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Sets or retrieves a configuration setting.
  *
  * @access   private
  * @param    string    configurationstring

  */
  private function cloudmail_office365_do_config($configurationstring) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $configurationstring . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 5, 1, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_office365->cast("char*", $param[1]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Creates a copy of a message.
  *
  * @access   private
  * @param    string    id

  * @param    string    destinationid

  */
  private function cloudmail_office365_do_copy($id, $destinationid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $destinationid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 6, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Copies a folder.
  *
  * @access   private
  * @param    string    id

  * @param    string    destfolderid

  */
  private function cloudmail_office365_do_copyfolder($id, $destfolderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $destfolderid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 7, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Creates a new category.
  *
  * @access   private
  * @param    string    displayname

  * @param    string    color

  */
  private function cloudmail_office365_do_createcategory($displayname, $color) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $displayname . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $color . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 8, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Creates a contact.
  *
  * @access   private
  * @param    string    firstname

  * @param    string    lastname

  * @param    string    emailaddress

  * @param    string    parentid

  */
  private function cloudmail_office365_do_createcontact($firstname, $lastname, $emailaddress, $parentid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 4 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 4 + 1 . "]", false, true);
    $cstr = $firstname . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $lastname . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $emailaddress . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $cstr = $parentid . "\0";
    $len = strlen((string)$cstr);
    $param[3] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[3], $cstr, $len);
    $cbparam[3] = 0;

    $param[4] = null;
    $cbparam[4] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 9, 4, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Creates a new email draft.
  *
  * @access   private
  * @param    int    drafttype

  * @param    string    id

  */
  private function cloudmail_office365_do_createdraft($drafttype, $id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $param[0] = $lib_cloudmail_office365->cast("void*", (int)$drafttype);
    $cbparam[0] = 0;

    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 10, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Creates a new folder.
  *
  * @access   private
  * @param    string    foldername

  * @param    string    parentfolderid

  */
  private function cloudmail_office365_do_createfolder($foldername, $parentfolderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $foldername . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $parentfolderid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 11, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Deletes a message.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_delete($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 12, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Deletes an attachment.
  *
  * @access   private
  * @param    string    id

  * @param    string    attachmentid

  */
  private function cloudmail_office365_do_deleteattachment($id, $attachmentid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $attachmentid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 13, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Deletes a mail category.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_deletecategory($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 14, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Deletes a contact.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_deletecontact($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 15, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Deletes a folder.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_deletefolder($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 16, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Forward a message.
  *
  * @access   private
  * @param    string    id

  * @param    string    messagehtml

  * @param    string    torecipients

  */
  private function cloudmail_office365_do_forward($id, $messagehtml, $torecipients) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 3 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 3 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $messagehtml . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $torecipients . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $param[3] = null;
    $cbparam[3] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 17, 3, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves a mail category.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_getcategory($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 18, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves the contact by Id.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_getcontact($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 19, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves the contact property value by JSONPath.
  *
  * @access   private
  * @param    int    index

  * @param    string    jsonpath

  */
  private function cloudmail_office365_do_getcontactfield($index, $jsonpath) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $param[0] = $lib_cloudmail_office365->cast("void*", (int)$index);
    $cbparam[0] = 0;

    $cstr = $jsonpath . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 20, 2, $param, $cbparam, null);

    if ((int)$retval) $retval = "";
    else $retval = FFI::string($lib_cloudmail_office365->cast("char*", $param[2]));

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves a folder.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_getfolder($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 21, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Interrupt the current method.
  *
  * @access   private
  */
  private function cloudmail_office365_do_interrupt() {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 22, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists all of a message's attachments.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_listattachments($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 23, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists all mail categories.
  *
  * @access   private
  */
  private function cloudmail_office365_do_listcategories() {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 24, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists messages that have been changed within a specified folder.
  *
  * @access   private
  * @param    string    id

  * @param    int    maxpagesize

  */
  private function cloudmail_office365_do_listchanges($id, $maxpagesize) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = $lib_cloudmail_office365->cast("void*", (int)$maxpagesize);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 25, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists the contact folders found in the parent folder.
  *
  * @access   private
  * @param    string    parentfolderid

  */
  private function cloudmail_office365_do_listcontactfolders($parentfolderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $parentfolderid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 26, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists the contacts in a folder.
  *
  * @access   private
  * @param    string    folderid

  */
  private function cloudmail_office365_do_listcontacts($folderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $folderid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 27, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists the folders found in the parent folder.
  *
  * @access   private
  * @param    string    parentfolderid

  */
  private function cloudmail_office365_do_listfolders($parentfolderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $parentfolderid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 28, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Lists the messages in a folder.
  *
  * @access   private
  * @param    string    folderid

  * @param    string    filter

  */
  private function cloudmail_office365_do_listmessages($folderid, $filter) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $folderid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $filter . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 29, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Moves a folder.
  *
  * @access   private
  * @param    string    id

  * @param    string    destfolderid

  */
  private function cloudmail_office365_do_movefolder($id, $destfolderid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $destfolderid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 30, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Moves a message.
  *
  * @access   private
  * @param    string    id

  * @param    string    destinationid

  */
  private function cloudmail_office365_do_movemessage($id, $destinationid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $destinationid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 31, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Renames a folder.
  *
  * @access   private
  * @param    string    folderid

  * @param    string    newname

  */
  private function cloudmail_office365_do_renamefolder($folderid, $newname) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $folderid . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $newname . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 32, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Reply to a message.
  *
  * @access   private
  * @param    string    id

  * @param    string    messagehtml

  */
  private function cloudmail_office365_do_reply($id, $messagehtml) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $messagehtml . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 33, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * ReplyAll to a message.
  *
  * @access   private
  * @param    string    id

  * @param    string    messagehtml

  */
  private function cloudmail_office365_do_replyall($id, $messagehtml) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $messagehtml . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 34, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * This method will reset the class.
  *
  * @access   private
  */
  private function cloudmail_office365_do_reset() {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 0 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 0 + 1 . "]", false, true);
    $param[0] = null;
    $cbparam[0] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 35, 0, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves a message attachment.
  *
  * @access   private
  * @param    string    id

  * @param    string    attachmentid

  */
  private function cloudmail_office365_do_retrieveattachment($id, $attachmentid) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $attachmentid . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 36, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves a message.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_retrievemessage($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 37, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Retrieves the raw message of the specified message ID.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_retrievemessageraw($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 38, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Search for messages.
  *
  * @access   private
  * @param    string    searchkey

  */
  private function cloudmail_office365_do_search($searchkey) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $searchkey . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 39, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Send a custom HTTP request.
  *
  * @access   private
  * @param    string    httpmethod

  * @param    string    url

  * @param    string    postdata

  */
  private function cloudmail_office365_do_sendcustomrequest($httpmethod, $url, $postdata) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 3 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 3 + 1 . "]", false, true);
    $cstr = $httpmethod . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $url . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $postdata . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $param[3] = null;
    $cbparam[3] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 40, 3, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Sends an existing draft.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_senddraft($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 41, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Sends a new email.
  *
  * @access   private
  * @param    boolean    savetosentitems

  */
  private function cloudmail_office365_do_sendmail($savetosentitems) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $param[0] = $lib_cloudmail_office365->cast("void*", (int)$savetosentitems);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 42, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Sets the contact field value by JSONPath.
  *
  * @access   private
  * @param    int    index

  * @param    string    jsonpath

  * @param    string    value

  * @param    int    valuetype

  */
  private function cloudmail_office365_do_setcontactfield($index, $jsonpath, $value, $valuetype) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 4 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 4 + 1 . "]", false, true);
    $param[0] = $lib_cloudmail_office365->cast("void*", (int)$index);
    $cbparam[0] = 0;

    $cstr = $jsonpath . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $cstr = $value . "\0";
    $len = strlen((string)$cstr);
    $param[2] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[2], $cstr, $len);
    $cbparam[2] = 0;

    $param[3] = $lib_cloudmail_office365->cast("void*", (int)$valuetype);
    $cbparam[3] = 0;

    $param[4] = null;
    $cbparam[4] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 44, 4, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Updates a message.
  *
  * @access   private
  * @param    string    id

  */
  private function cloudmail_office365_do_update($id) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 45, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Updates a category.
  *
  * @access   private
  * @param    string    id

  * @param    string    color

  */
  private function cloudmail_office365_do_updatecategory($id, $color) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 2 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 2 + 1 . "]", false, true);
    $cstr = $id . "\0";
    $len = strlen((string)$cstr);
    $param[0] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[0], $cstr, $len);
    $cbparam[0] = 0;

    $cstr = $color . "\0";
    $len = strlen((string)$cstr);
    $param[1] = $lib_cloudmail_office365->new("char[" . $len . "]", false, true);
    FFI::memcpy($param[1], $cstr, $len);
    $cbparam[1] = 0;

    $param[2] = null;
    $cbparam[2] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 46, 2, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

 /**
  * Updates a contact.
  *
  * @access   private
  * @param    int    index

  */
  private function cloudmail_office365_do_updatecontact($index) {
    global $lib_cloudmail_office365;
    $param = $lib_cloudmail_office365->new("void*[" . 1 + 1 . "]", false, true);
    $cbparam = $lib_cloudmail_office365->new("int[" . 1 + 1 . "]", false, true);
    $param[0] = $lib_cloudmail_office365->cast("void*", (int)$index);
    $cbparam[0] = 0;

    $param[1] = null;
    $cbparam[1] = 0;

    $retval = $lib_cloudmail_office365->CloudMail_Office365_Do($this->handle, 47, 1, $param, $cbparam, null);

    $retval = (int)$retval;

//free(todo)
    FFI::free($cbparam);
    return $retval;
  }

  
  private function cloudmail_office365_get($pid, $index = 0) {
    global $lib_cloudmail_office365;

    switch ($pid) {
      case 0:
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 0, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
      case 1: { /* ATTACHMENTCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 1, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 2: { /* ATTACHMENTATTACHMENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 2, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 3: { /* ATTACHMENTCONTENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 3, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 4: { /* ATTACHMENTCONTENTLOCATION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 4, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 5: { /* ATTACHMENTCONTENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 5, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 6: { /* ATTACHMENTDATA */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 6, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 7: { /* ATTACHMENTFILENAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 7, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 8: { /* ATTACHMENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 8, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 10: { /* ATTACHMENTISINLINE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 10, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 11: { /* ATTACHMENTLASTMODIFIEDDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 11, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 12: { /* ATTACHMENTNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 12, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 13: { /* ATTACHMENTPARENTINDEX */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 13, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 14: { /* ATTACHMENTSIZE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 14, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 15: { /* AUTHORIZATION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 15, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 16: { /* CATEGORYCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 16, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 17: { /* CATEGORYCOLOR */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 17, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 18: { /* CATEGORYDISPLAYNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 18, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 19: { /* CATEGORYID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 19, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 20: { /* CHANGEMARKER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 20, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 21: { /* CONTACTFOLDERCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 21, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 22: { /* CONTACTFOLDERID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 22, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 23: { /* CONTACTFOLDERNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 23, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 24: { /* CONTACTFOLDERPARENTFOLDERID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 24, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 25: { /* CONTACTSCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 25, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 26: { /* CONTACTCOMPANYNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 26, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 27: { /* CONTACTDISPLAYNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 27, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 28: { /* CONTACTEMAILADDRESS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 28, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 29: { /* CONTACTEMAILADDRESSESCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 29, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 30: { /* CONTACTEMAILADDRESSINDEX */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 30, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 31: { /* CONTACTFIRSTNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 31, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 32: { /* CONTACTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 32, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 33: { /* CONTACTJSON */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 33, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 34: { /* CONTACTLASTNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 34, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 35: { /* CONTACTNOTES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 35, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 36: { /* CONTACTPARENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 36, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 37: { /* CONTACTPHONEINDEX */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 37, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 38: { /* CONTACTPHONENUMBER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 38, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 39: { /* CONTACTPHONESCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 39, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 40: { /* CONTACTPHONETYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 40, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 41: { /* FIREWALLAUTODETECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 41, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 42: { /* FIREWALLTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 42, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 43: { /* FIREWALLHOST */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 43, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 44: { /* FIREWALLPASSWORD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 44, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 45: { /* FIREWALLPORT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 45, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 46: { /* FIREWALLUSER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 46, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 47: { /* FOLDERCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 47, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 48: { /* FOLDERCHILDFOLDERCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 48, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 49: { /* FOLDERCHILDFOLDERS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 49, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 50: { /* FOLDERDISPLAYNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 50, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 51: { /* FOLDERID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 51, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 52: { /* FOLDERMESSAGERULES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 52, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 53: { /* FOLDERMESSAGES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 53, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 54: { /* FOLDERMULTIVALUEEXTENDEDPROPERTIES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 54, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 55: { /* FOLDERPARENTFOLDERID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 55, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 56: { /* FOLDERSINGLEVALUEEXTENDEDPROPERTIES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 56, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 57: { /* FOLDERTOTALITEMCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 57, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 58: { /* FOLDERUNREADITEMCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 58, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 59: { /* LISTCONTACTSMARKER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 59, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 60: { /* LISTFOLDERSMARKER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 60, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 61: { /* LISTMESSAGESMARKER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 61, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 62: { /* MESSAGE */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 62, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 63: { /* MESSAGEATTACHMENTCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 63, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 64: { /* MESSAGEATTACHMENTATTACHMENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 64, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 65: { /* MESSAGEATTACHMENTCONTENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 65, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 66: { /* MESSAGEATTACHMENTCONTENTLOCATION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 66, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 67: { /* MESSAGEATTACHMENTCONTENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 67, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 68: { /* MESSAGEATTACHMENTDATA */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 68, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 69: { /* MESSAGEATTACHMENTFILENAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 69, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 70: { /* MESSAGEATTACHMENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 70, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 72: { /* MESSAGEATTACHMENTISINLINE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 72, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 73: { /* MESSAGEATTACHMENTLASTMODIFIEDDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 73, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 74: { /* MESSAGEATTACHMENTNAME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 74, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 75: { /* MESSAGEATTACHMENTPARENTINDEX */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 75, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 76: { /* MESSAGEATTACHMENTSIZE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 76, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 77: { /* MESSAGEBCC */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 77, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 78: { /* MESSAGEBODYCONTENT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 78, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 79: { /* MESSAGEBODYCONTENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 79, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 80: { /* MESSAGECC */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 80, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 81: { /* MESSAGEDELIVERYRECEIPT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 81, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 82: { /* MESSAGEFROM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 82, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 83: { /* MESSAGEIMPORTANCE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 83, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 84: { /* MESSAGEINFOCOUNT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 84, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 85: { /* MESSAGEINFOBCC */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 85, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 86: { /* MESSAGEINFOBODYCONTENT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 86, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 87: { /* MESSAGEINFOBODYCONTENTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 87, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 88: { /* MESSAGEINFOBODYPREVIEW */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 88, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 89: { /* MESSAGEINFOCATEGORIES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 89, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 90: { /* MESSAGEINFOCC */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 90, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 91: { /* MESSAGEINFOCONVERSATIONID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 91, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 92: { /* MESSAGEINFOCONVERSATIONINDEX */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 92, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 93: { /* MESSAGEINFOCREATEDDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 93, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 94: { /* MESSAGEINFODELIVERYRECEIPTREQUESTED */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 94, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 95: { /* MESSAGEINFOFLAGSTATUS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 95, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 96: { /* MESSAGEINFOFROM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 96, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 97: { /* MESSAGEINFOHASATTACHMENTS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 97, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 98: { /* MESSAGEINFOID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 98, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 99: { /* MESSAGEINFOIMPORTANCE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 99, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 100: { /* MESSAGEINFOINFERENCECLASSIFICATION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 100, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 101: { /* MESSAGEINFOISDRAFT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 101, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 102: { /* MESSAGEINFOISREAD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 102, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 103: { /* MESSAGEINFOJSON */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 103, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 104: { /* MESSAGEINFOLASTMODIFIEDDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 104, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 105: { /* MESSAGEINFOMESSAGEHEADERS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 105, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 106: { /* MESSAGEINFOMESSAGEID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 106, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 107: { /* MESSAGEINFOPARENTFOLDERID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 107, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 108: { /* MESSAGEINFOREADRECEIPTREQUESTED */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 108, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 109: { /* MESSAGEINFORECEIVEDDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 109, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 110: { /* MESSAGEINFOREPLYTO */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 110, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 111: { /* MESSAGEINFOSENDER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 111, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 112: { /* MESSAGEINFOSENTDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 112, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 113: { /* MESSAGEINFOSUBJECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 113, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 114: { /* MESSAGEINFOTO */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 114, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 115: { /* MESSAGEINFOWEBLINK */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 115, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 116: { /* MESSAGEOTHERHEADERS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 116, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 117: { /* MESSAGEREADRECEIPT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 117, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 118: { /* MESSAGEREPLYTO */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 118, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 119: { /* MESSAGESUBJECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 119, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 120: { /* MESSAGETO */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 120, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 121: { /* NEXTCHANGEMARKER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 121, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 122: { /* OAUTHACCESSTOKEN */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 122, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 123: { /* OAUTHAUTHORIZATIONCODE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 123, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 124: { /* OAUTHAUTHORIZATIONSCOPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 124, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 125: { /* OAUTHCLIENTID */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 125, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 126: { /* OAUTHCLIENTPROFILE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 126, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 127: { /* OAUTHCLIENTSECRET */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 127, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 128: { /* OAUTHGRANTTYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 128, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 129: { /* OAUTHREFRESHTOKEN */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 129, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 130: { /* OAUTHREQUESTREFRESHTOKEN */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 130, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 131: { /* OAUTHRETURNURL */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 131, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 132: { /* OAUTHSERVERAUTHURL */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 132, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 133: { /* OAUTHSERVERTOKENURL */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 133, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 134: { /* OAUTHWEBAUTHURL */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 134, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 135: { /* PROXYAUTHSCHEME */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 135, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 136: { /* PROXYAUTODETECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 136, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 137: { /* PROXYPASSWORD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 137, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 138: { /* PROXYPORT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 138, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 139: { /* PROXYSERVER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 139, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 140: { /* PROXYSSL */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 140, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 141: { /* PROXYUSER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 141, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 142: { /* SELECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 142, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 143: { /* SSLACCEPTSERVERCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 143, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 144: { /* SSLACCEPTSERVERCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 144, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 145: { /* SSLACCEPTSERVERCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 145, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 146: { /* SSLACCEPTSERVERCERTFINGERPRINT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 146, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 147: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 147, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 148: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 148, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 149: { /* SSLACCEPTSERVERCERTISSUER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 149, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 150: { /* SSLACCEPTSERVERCERTPRIVATEKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 150, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 151: { /* SSLACCEPTSERVERCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 151, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 152: { /* SSLACCEPTSERVERCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 152, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 153: { /* SSLACCEPTSERVERCERTPUBLICKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 153, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 154: { /* SSLACCEPTSERVERCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 154, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 155: { /* SSLACCEPTSERVERCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 155, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 156: { /* SSLACCEPTSERVERCERTSERIALNUMBER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 156, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 157: { /* SSLACCEPTSERVERCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 157, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 158: { /* SSLACCEPTSERVERCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 158, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 159: { /* SSLACCEPTSERVERCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 159, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 160: { /* SSLACCEPTSERVERCERTSTORETYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 160, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 161: { /* SSLACCEPTSERVERCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 161, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 162: { /* SSLACCEPTSERVERCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 162, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 163: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 163, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 164: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 164, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 165: { /* SSLACCEPTSERVERCERTUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 165, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 166: { /* SSLACCEPTSERVERCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 166, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 167: { /* SSLACCEPTSERVERCERTVERSION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 167, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 168: { /* SSLACCEPTSERVERCERTSUBJECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 168, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 169: { /* SSLACCEPTSERVERCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 169, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 170: { /* SSLCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 170, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 171: { /* SSLCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 171, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 172: { /* SSLCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 172, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 173: { /* SSLCERTFINGERPRINT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 173, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 174: { /* SSLCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 174, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 175: { /* SSLCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 175, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 176: { /* SSLCERTISSUER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 176, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 177: { /* SSLCERTPRIVATEKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 177, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 178: { /* SSLCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 178, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 179: { /* SSLCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 179, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 180: { /* SSLCERTPUBLICKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 180, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 181: { /* SSLCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 181, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 182: { /* SSLCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 182, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 183: { /* SSLCERTSERIALNUMBER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 183, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 184: { /* SSLCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 184, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 185: { /* SSLCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 185, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 186: { /* SSLCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 186, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 187: { /* SSLCERTSTORETYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 187, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 188: { /* SSLCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 188, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 189: { /* SSLCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 189, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 190: { /* SSLCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 190, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 191: { /* SSLCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 191, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 192: { /* SSLCERTUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 192, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 193: { /* SSLCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 193, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 194: { /* SSLCERTVERSION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 194, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 195: { /* SSLCERTSUBJECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 195, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 196: { /* SSLCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 196, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 197: { /* SSLPROVIDER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 197, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 198: { /* SSLSERVERCERTEFFECTIVEDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 198, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 199: { /* SSLSERVERCERTEXPIRATIONDATE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 199, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 200: { /* SSLSERVERCERTEXTENDEDKEYUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 200, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 201: { /* SSLSERVERCERTFINGERPRINT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 201, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 202: { /* SSLSERVERCERTFINGERPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 202, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 203: { /* SSLSERVERCERTFINGERPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 203, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 204: { /* SSLSERVERCERTISSUER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 204, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 205: { /* SSLSERVERCERTPRIVATEKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 205, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 206: { /* SSLSERVERCERTPRIVATEKEYAVAILABLE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 206, (int)$index, null, null);
        return isset($ret) ? true : false;
        break;
      }

      case 207: { /* SSLSERVERCERTPRIVATEKEYCONTAINER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 207, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 208: { /* SSLSERVERCERTPUBLICKEY */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 208, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 209: { /* SSLSERVERCERTPUBLICKEYALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 209, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 210: { /* SSLSERVERCERTPUBLICKEYLENGTH */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 210, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 211: { /* SSLSERVERCERTSERIALNUMBER */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 211, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 212: { /* SSLSERVERCERTSIGNATUREALGORITHM */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 212, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 213: { /* SSLSERVERCERTSTORE */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 213, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 214: { /* SSLSERVERCERTSTOREPASSWORD */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 214, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 215: { /* SSLSERVERCERTSTORETYPE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 215, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 216: { /* SSLSERVERCERTSUBJECTALTNAMES */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 216, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 217: { /* SSLSERVERCERTTHUMBPRINTMD5 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 217, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 218: { /* SSLSERVERCERTTHUMBPRINTSHA1 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 218, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 219: { /* SSLSERVERCERTTHUMBPRINTSHA256 */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 219, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 220: { /* SSLSERVERCERTUSAGE */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 220, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 221: { /* SSLSERVERCERTUSAGEFLAGS */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 221, (int)$index, null, null);
        if (isset($ret)) {
          return $ret - $lib_cloudmail_office365->cast('void*', 0);
        } else {
          return 0;
        }
        break;
      }

      case 222: { /* SSLSERVERCERTVERSION */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 222, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 223: { /* SSLSERVERCERTSUBJECT */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 223, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }

      case 224: { /* SSLSERVERCERTENCODED */
        $blen = FFI::addr($lib_cloudmail_office365->new("int"));
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 224, (int)$index, $blen, null);
        if (isset($ret) && $blen[0]) return FFI::string($lib_cloudmail_office365->cast("char*", $ret), $blen[0]);
        break;
      }

      case 8001/*PID_RUNTIME_LICENSE*/: { /* RuntimeLicense */
        $ret = $lib_cloudmail_office365->CloudMail_Office365_Get($this->handle, 8001, (int)$index, null, null);
        if (isset($ret)) return FFI::string($lib_cloudmail_office365->cast("char*", $ret));
        break;
      }
      default: {
        php_error(E_ERROR, "invalid property id");
        break;
      }
    }

    return -1;
  }

  private function cloudmail_office365_set($pid, $value, $index = 0) {
    global $lib_cloudmail_office365;
    $retval = -1;

    switch ($pid) {

      case 1: { /* ATTACHMENTCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 1, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for AttachmentCount.", 0);
        }
        break;
      }


      case 2: { /* ATTACHMENTATTACHMENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 2, (int)$index, (string)$value, 0);
        break;
      }


      case 3: { /* ATTACHMENTCONTENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 3, (int)$index, (string)$value, 0);
        break;
      }


      case 4: { /* ATTACHMENTCONTENTLOCATION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 4, (int)$index, (string)$value, 0);
        break;
      }


      case 5: { /* ATTACHMENTCONTENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 5, (int)$index, (string)$value, 0);
        break;
      }


      case 6: { /* ATTACHMENTDATA */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 6, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 7: { /* ATTACHMENTFILENAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 7, (int)$index, (string)$value, 0);
        break;
      }


      case 8: { /* ATTACHMENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 8, (int)$index, (string)$value, 0);
        break;
      }


      case 10: { /* ATTACHMENTISINLINE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 10, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for AttachmentIsInline.", 0);
        }
        break;
      }


      case 11: { /* ATTACHMENTLASTMODIFIEDDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 11, (int)$index, (string)$value, 0);
        break;
      }


      case 12: { /* ATTACHMENTNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 12, (int)$index, (string)$value, 0);
        break;
      }


      case 13: { /* ATTACHMENTPARENTINDEX */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 13, (int)$index, (string)$value, 0);
        break;
      }


      case 14: { /* ATTACHMENTSIZE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 14, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for AttachmentSize.", 0);
        }
        break;
      }


      case 15: { /* AUTHORIZATION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 15, (int)$index, (string)$value, 0);
        break;
      }


      case 16: { /* CATEGORYCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 16, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for CategoryCount.", 0);
        }
        break;
      }


      case 17: { /* CATEGORYCOLOR */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 17, (int)$index, (string)$value, 0);
        break;
      }


      case 18: { /* CATEGORYDISPLAYNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 18, (int)$index, (string)$value, 0);
        break;
      }


      case 19: { /* CATEGORYID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 19, (int)$index, (string)$value, 0);
        break;
      }


      case 20: { /* CHANGEMARKER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 20, (int)$index, (string)$value, 0);
        break;
      }


      case 21: { /* CONTACTFOLDERCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 21, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactFolderCount.", 0);
        }
        break;
      }


      case 22: { /* CONTACTFOLDERID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 22, (int)$index, (string)$value, 0);
        break;
      }


      case 23: { /* CONTACTFOLDERNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 23, (int)$index, (string)$value, 0);
        break;
      }


      case 24: { /* CONTACTFOLDERPARENTFOLDERID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 24, (int)$index, (string)$value, 0);
        break;
      }


      case 25: { /* CONTACTSCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 25, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactsCount.", 0);
        }
        break;
      }


      case 26: { /* CONTACTCOMPANYNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 26, (int)$index, (string)$value, 0);
        break;
      }


      case 27: { /* CONTACTDISPLAYNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 27, (int)$index, (string)$value, 0);
        break;
      }


      case 28: { /* CONTACTEMAILADDRESS */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 28, (int)$index, (string)$value, 0);
        break;
      }


      case 29: { /* CONTACTEMAILADDRESSESCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 29, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactEmailAddressesCount.", 0);
        }
        break;
      }


      case 30: { /* CONTACTEMAILADDRESSINDEX */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 30, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactEmailAddressIndex.", 0);
        }
        break;
      }


      case 31: { /* CONTACTFIRSTNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 31, (int)$index, (string)$value, 0);
        break;
      }


      case 32: { /* CONTACTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 32, (int)$index, (string)$value, 0);
        break;
      }


      case 33: { /* CONTACTJSON */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 33, (int)$index, (string)$value, 0);
        break;
      }


      case 34: { /* CONTACTLASTNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 34, (int)$index, (string)$value, 0);
        break;
      }


      case 35: { /* CONTACTNOTES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 35, (int)$index, (string)$value, 0);
        break;
      }


      case 36: { /* CONTACTPARENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 36, (int)$index, (string)$value, 0);
        break;
      }


      case 37: { /* CONTACTPHONEINDEX */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 37, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactPhoneIndex.", 0);
        }
        break;
      }


      case 38: { /* CONTACTPHONENUMBER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 38, (int)$index, (string)$value, 0);
        break;
      }


      case 39: { /* CONTACTPHONESCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 39, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactPhonesCount.", 0);
        }
        break;
      }


      case 40: { /* CONTACTPHONETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 40, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ContactPhoneType.", 0);
        }
        break;
      }


      case 41: { /* FIREWALLAUTODETECT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 41, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallAutoDetect.", 0);
        }
        break;
      }


      case 42: { /* FIREWALLTYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 42, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallType.", 0);
        }
        break;
      }


      case 43: { /* FIREWALLHOST */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 43, (int)$index, (string)$value, 0);
        break;
      }


      case 44: { /* FIREWALLPASSWORD */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 44, (int)$index, (string)$value, 0);
        break;
      }


      case 45: { /* FIREWALLPORT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 45, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FirewallPort.", 0);
        }
        break;
      }


      case 46: { /* FIREWALLUSER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 46, (int)$index, (string)$value, 0);
        break;
      }


      case 47: { /* FOLDERCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 47, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FolderCount.", 0);
        }
        break;
      }


      case 48: { /* FOLDERCHILDFOLDERCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 48, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FolderChildFolderCount.", 0);
        }
        break;
      }


      case 49: { /* FOLDERCHILDFOLDERS */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 49, (int)$index, (string)$value, 0);
        break;
      }


      case 50: { /* FOLDERDISPLAYNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 50, (int)$index, (string)$value, 0);
        break;
      }


      case 51: { /* FOLDERID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 51, (int)$index, (string)$value, 0);
        break;
      }


      case 52: { /* FOLDERMESSAGERULES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 52, (int)$index, (string)$value, 0);
        break;
      }


      case 53: { /* FOLDERMESSAGES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 53, (int)$index, (string)$value, 0);
        break;
      }


      case 54: { /* FOLDERMULTIVALUEEXTENDEDPROPERTIES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 54, (int)$index, (string)$value, 0);
        break;
      }


      case 55: { /* FOLDERPARENTFOLDERID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 55, (int)$index, (string)$value, 0);
        break;
      }


      case 56: { /* FOLDERSINGLEVALUEEXTENDEDPROPERTIES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 56, (int)$index, (string)$value, 0);
        break;
      }


      case 57: { /* FOLDERTOTALITEMCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 57, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FolderTotalItemCount.", 0);
        }
        break;
      }


      case 58: { /* FOLDERUNREADITEMCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 58, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for FolderUnreadItemCount.", 0);
        }
        break;
      }


      case 59: { /* LISTCONTACTSMARKER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 59, (int)$index, (string)$value, 0);
        break;
      }


      case 60: { /* LISTFOLDERSMARKER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 60, (int)$index, (string)$value, 0);
        break;
      }


      case 61: { /* LISTMESSAGESMARKER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 61, (int)$index, (string)$value, 0);
        break;
      }


      case 62: { /* MESSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 62, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 63: { /* MESSAGEATTACHMENTCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 63, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageAttachmentCount.", 0);
        }
        break;
      }


      case 64: { /* MESSAGEATTACHMENTATTACHMENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 64, (int)$index, (string)$value, 0);
        break;
      }


      case 65: { /* MESSAGEATTACHMENTCONTENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 65, (int)$index, (string)$value, 0);
        break;
      }


      case 66: { /* MESSAGEATTACHMENTCONTENTLOCATION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 66, (int)$index, (string)$value, 0);
        break;
      }


      case 67: { /* MESSAGEATTACHMENTCONTENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 67, (int)$index, (string)$value, 0);
        break;
      }


      case 68: { /* MESSAGEATTACHMENTDATA */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 68, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 69: { /* MESSAGEATTACHMENTFILENAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 69, (int)$index, (string)$value, 0);
        break;
      }


      case 70: { /* MESSAGEATTACHMENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 70, (int)$index, (string)$value, 0);
        break;
      }


      case 72: { /* MESSAGEATTACHMENTISINLINE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 72, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageAttachmentIsInline.", 0);
        }
        break;
      }


      case 73: { /* MESSAGEATTACHMENTLASTMODIFIEDDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 73, (int)$index, (string)$value, 0);
        break;
      }


      case 74: { /* MESSAGEATTACHMENTNAME */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 74, (int)$index, (string)$value, 0);
        break;
      }


      case 75: { /* MESSAGEATTACHMENTPARENTINDEX */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 75, (int)$index, (string)$value, 0);
        break;
      }


      case 76: { /* MESSAGEATTACHMENTSIZE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 76, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageAttachmentSize.", 0);
        }
        break;
      }


      case 77: { /* MESSAGEBCC */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 77, (int)$index, (string)$value, 0);
        break;
      }


      case 78: { /* MESSAGEBODYCONTENT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 78, (int)$index, (string)$value, 0);
        break;
      }


      case 79: { /* MESSAGEBODYCONTENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 79, (int)$index, (string)$value, 0);
        break;
      }


      case 80: { /* MESSAGECC */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 80, (int)$index, (string)$value, 0);
        break;
      }


      case 81: { /* MESSAGEDELIVERYRECEIPT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 81, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageDeliveryReceipt.", 0);
        }
        break;
      }


      case 82: { /* MESSAGEFROM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 82, (int)$index, (string)$value, 0);
        break;
      }


      case 83: { /* MESSAGEIMPORTANCE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 83, (int)$index, (string)$value, 0);
        break;
      }


      case 84: { /* MESSAGEINFOCOUNT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 84, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoCount.", 0);
        }
        break;
      }


      case 85: { /* MESSAGEINFOBCC */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 85, (int)$index, (string)$value, 0);
        break;
      }


      case 86: { /* MESSAGEINFOBODYCONTENT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 86, (int)$index, (string)$value, 0);
        break;
      }


      case 87: { /* MESSAGEINFOBODYCONTENTTYPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 87, (int)$index, (string)$value, 0);
        break;
      }


      case 88: { /* MESSAGEINFOBODYPREVIEW */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 88, (int)$index, (string)$value, 0);
        break;
      }


      case 89: { /* MESSAGEINFOCATEGORIES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 89, (int)$index, (string)$value, 0);
        break;
      }


      case 90: { /* MESSAGEINFOCC */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 90, (int)$index, (string)$value, 0);
        break;
      }


      case 91: { /* MESSAGEINFOCONVERSATIONID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 91, (int)$index, (string)$value, 0);
        break;
      }


      case 92: { /* MESSAGEINFOCONVERSATIONINDEX */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 92, (int)$index, (string)$value, 0);
        break;
      }


      case 93: { /* MESSAGEINFOCREATEDDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 93, (int)$index, (string)$value, 0);
        break;
      }


      case 94: { /* MESSAGEINFODELIVERYRECEIPTREQUESTED */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 94, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoDeliveryReceiptRequested.", 0);
        }
        break;
      }


      case 95: { /* MESSAGEINFOFLAGSTATUS */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 95, (int)$index, (string)$value, 0);
        break;
      }


      case 96: { /* MESSAGEINFOFROM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 96, (int)$index, (string)$value, 0);
        break;
      }


      case 97: { /* MESSAGEINFOHASATTACHMENTS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 97, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoHasAttachments.", 0);
        }
        break;
      }


      case 98: { /* MESSAGEINFOID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 98, (int)$index, (string)$value, 0);
        break;
      }


      case 99: { /* MESSAGEINFOIMPORTANCE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 99, (int)$index, (string)$value, 0);
        break;
      }


      case 100: { /* MESSAGEINFOINFERENCECLASSIFICATION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 100, (int)$index, (string)$value, 0);
        break;
      }


      case 101: { /* MESSAGEINFOISDRAFT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 101, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoIsDraft.", 0);
        }
        break;
      }


      case 102: { /* MESSAGEINFOISREAD */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 102, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoIsRead.", 0);
        }
        break;
      }


      case 103: { /* MESSAGEINFOJSON */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 103, (int)$index, (string)$value, 0);
        break;
      }


      case 104: { /* MESSAGEINFOLASTMODIFIEDDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 104, (int)$index, (string)$value, 0);
        break;
      }


      case 105: { /* MESSAGEINFOMESSAGEHEADERS */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 105, (int)$index, (string)$value, 0);
        break;
      }


      case 106: { /* MESSAGEINFOMESSAGEID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 106, (int)$index, (string)$value, 0);
        break;
      }


      case 107: { /* MESSAGEINFOPARENTFOLDERID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 107, (int)$index, (string)$value, 0);
        break;
      }


      case 108: { /* MESSAGEINFOREADRECEIPTREQUESTED */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 108, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageInfoReadReceiptRequested.", 0);
        }
        break;
      }


      case 109: { /* MESSAGEINFORECEIVEDDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 109, (int)$index, (string)$value, 0);
        break;
      }


      case 110: { /* MESSAGEINFOREPLYTO */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 110, (int)$index, (string)$value, 0);
        break;
      }


      case 111: { /* MESSAGEINFOSENDER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 111, (int)$index, (string)$value, 0);
        break;
      }


      case 112: { /* MESSAGEINFOSENTDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 112, (int)$index, (string)$value, 0);
        break;
      }


      case 113: { /* MESSAGEINFOSUBJECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 113, (int)$index, (string)$value, 0);
        break;
      }


      case 114: { /* MESSAGEINFOTO */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 114, (int)$index, (string)$value, 0);
        break;
      }


      case 115: { /* MESSAGEINFOWEBLINK */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 115, (int)$index, (string)$value, 0);
        break;
      }


      case 116: { /* MESSAGEOTHERHEADERS */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 116, (int)$index, (string)$value, 0);
        break;
      }


      case 117: { /* MESSAGEREADRECEIPT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 117, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for MessageReadReceipt.", 0);
        }
        break;
      }


      case 118: { /* MESSAGEREPLYTO */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 118, (int)$index, (string)$value, 0);
        break;
      }


      case 119: { /* MESSAGESUBJECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 119, (int)$index, (string)$value, 0);
        break;
      }


      case 120: { /* MESSAGETO */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 120, (int)$index, (string)$value, 0);
        break;
      }


      case 121: { /* NEXTCHANGEMARKER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 121, (int)$index, (string)$value, 0);
        break;
      }


      case 122: { /* OAUTHACCESSTOKEN */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 122, (int)$index, (string)$value, 0);
        break;
      }


      case 123: { /* OAUTHAUTHORIZATIONCODE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 123, (int)$index, (string)$value, 0);
        break;
      }


      case 124: { /* OAUTHAUTHORIZATIONSCOPE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 124, (int)$index, (string)$value, 0);
        break;
      }


      case 125: { /* OAUTHCLIENTID */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 125, (int)$index, (string)$value, 0);
        break;
      }


      case 126: { /* OAUTHCLIENTPROFILE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 126, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for OAuthClientProfile.", 0);
        }
        break;
      }


      case 127: { /* OAUTHCLIENTSECRET */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 127, (int)$index, (string)$value, 0);
        break;
      }


      case 128: { /* OAUTHGRANTTYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 128, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for OAuthGrantType.", 0);
        }
        break;
      }


      case 129: { /* OAUTHREFRESHTOKEN */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 129, (int)$index, (string)$value, 0);
        break;
      }


      case 130: { /* OAUTHREQUESTREFRESHTOKEN */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 130, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for OAuthRequestRefreshToken.", 0);
        }
        break;
      }


      case 131: { /* OAUTHRETURNURL */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 131, (int)$index, (string)$value, 0);
        break;
      }


      case 132: { /* OAUTHSERVERAUTHURL */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 132, (int)$index, (string)$value, 0);
        break;
      }


      case 133: { /* OAUTHSERVERTOKENURL */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 133, (int)$index, (string)$value, 0);
        break;
      }


      case 134: { /* OAUTHWEBAUTHURL */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 134, (int)$index, (string)$value, 0);
        break;
      }


      case 135: { /* PROXYAUTHSCHEME */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 135, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyAuthScheme.", 0);
        }
        break;
      }


      case 136: { /* PROXYAUTODETECT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 136, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyAutoDetect.", 0);
        }
        break;
      }


      case 137: { /* PROXYPASSWORD */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 137, (int)$index, (string)$value, 0);
        break;
      }


      case 138: { /* PROXYPORT */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 138, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxyPort.", 0);
        }
        break;
      }


      case 139: { /* PROXYSERVER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 139, (int)$index, (string)$value, 0);
        break;
      }


      case 140: { /* PROXYSSL */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 140, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for ProxySSL.", 0);
        }
        break;
      }


      case 141: { /* PROXYUSER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 141, (int)$index, (string)$value, 0);
        break;
      }


      case 142: { /* SELECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 142, (int)$index, (string)$value, 0);
        break;
      }


      case 143: { /* SSLACCEPTSERVERCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 143, (int)$index, (string)$value, 0);
        break;
      }


      case 144: { /* SSLACCEPTSERVERCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 144, (int)$index, (string)$value, 0);
        break;
      }


      case 145: { /* SSLACCEPTSERVERCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 145, (int)$index, (string)$value, 0);
        break;
      }


      case 146: { /* SSLACCEPTSERVERCERTFINGERPRINT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 146, (int)$index, (string)$value, 0);
        break;
      }


      case 147: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 147, (int)$index, (string)$value, 0);
        break;
      }


      case 148: { /* SSLACCEPTSERVERCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 148, (int)$index, (string)$value, 0);
        break;
      }


      case 149: { /* SSLACCEPTSERVERCERTISSUER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 149, (int)$index, (string)$value, 0);
        break;
      }


      case 150: { /* SSLACCEPTSERVERCERTPRIVATEKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 150, (int)$index, (string)$value, 0);
        break;
      }


      case 151: { /* SSLACCEPTSERVERCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 151, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 152: { /* SSLACCEPTSERVERCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 152, (int)$index, (string)$value, 0);
        break;
      }


      case 153: { /* SSLACCEPTSERVERCERTPUBLICKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 153, (int)$index, (string)$value, 0);
        break;
      }


      case 154: { /* SSLACCEPTSERVERCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 154, (int)$index, (string)$value, 0);
        break;
      }


      case 155: { /* SSLACCEPTSERVERCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 155, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertPublicKeyLength.", 0);
        }
        break;
      }


      case 156: { /* SSLACCEPTSERVERCERTSERIALNUMBER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 156, (int)$index, (string)$value, 0);
        break;
      }


      case 157: { /* SSLACCEPTSERVERCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 157, (int)$index, (string)$value, 0);
        break;
      }


      case 158: { /* SSLACCEPTSERVERCERTSTORE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 158, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 159: { /* SSLACCEPTSERVERCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 159, (int)$index, (string)$value, 0);
        break;
      }


      case 160: { /* SSLACCEPTSERVERCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 160, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertStoreType.", 0);
        }
        break;
      }


      case 161: { /* SSLACCEPTSERVERCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 161, (int)$index, (string)$value, 0);
        break;
      }


      case 162: { /* SSLACCEPTSERVERCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 162, (int)$index, (string)$value, 0);
        break;
      }


      case 163: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 163, (int)$index, (string)$value, 0);
        break;
      }


      case 164: { /* SSLACCEPTSERVERCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 164, (int)$index, (string)$value, 0);
        break;
      }


      case 165: { /* SSLACCEPTSERVERCERTUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 165, (int)$index, (string)$value, 0);
        break;
      }


      case 166: { /* SSLACCEPTSERVERCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 166, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLAcceptServerCertUsageFlags.", 0);
        }
        break;
      }


      case 167: { /* SSLACCEPTSERVERCERTVERSION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 167, (int)$index, (string)$value, 0);
        break;
      }


      case 168: { /* SSLACCEPTSERVERCERTSUBJECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 168, (int)$index, (string)$value, 0);
        break;
      }


      case 169: { /* SSLACCEPTSERVERCERTENCODED */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 169, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 170: { /* SSLCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 170, (int)$index, (string)$value, 0);
        break;
      }


      case 171: { /* SSLCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 171, (int)$index, (string)$value, 0);
        break;
      }


      case 172: { /* SSLCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 172, (int)$index, (string)$value, 0);
        break;
      }


      case 173: { /* SSLCERTFINGERPRINT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 173, (int)$index, (string)$value, 0);
        break;
      }


      case 174: { /* SSLCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 174, (int)$index, (string)$value, 0);
        break;
      }


      case 175: { /* SSLCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 175, (int)$index, (string)$value, 0);
        break;
      }


      case 176: { /* SSLCERTISSUER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 176, (int)$index, (string)$value, 0);
        break;
      }


      case 177: { /* SSLCERTPRIVATEKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 177, (int)$index, (string)$value, 0);
        break;
      }


      case 178: { /* SSLCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 178, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 179: { /* SSLCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 179, (int)$index, (string)$value, 0);
        break;
      }


      case 180: { /* SSLCERTPUBLICKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 180, (int)$index, (string)$value, 0);
        break;
      }


      case 181: { /* SSLCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 181, (int)$index, (string)$value, 0);
        break;
      }


      case 182: { /* SSLCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 182, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertPublicKeyLength.", 0);
        }
        break;
      }


      case 183: { /* SSLCERTSERIALNUMBER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 183, (int)$index, (string)$value, 0);
        break;
      }


      case 184: { /* SSLCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 184, (int)$index, (string)$value, 0);
        break;
      }


      case 185: { /* SSLCERTSTORE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 185, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 186: { /* SSLCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 186, (int)$index, (string)$value, 0);
        break;
      }


      case 187: { /* SSLCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 187, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertStoreType.", 0);
        }
        break;
      }


      case 188: { /* SSLCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 188, (int)$index, (string)$value, 0);
        break;
      }


      case 189: { /* SSLCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 189, (int)$index, (string)$value, 0);
        break;
      }


      case 190: { /* SSLCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 190, (int)$index, (string)$value, 0);
        break;
      }


      case 191: { /* SSLCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 191, (int)$index, (string)$value, 0);
        break;
      }


      case 192: { /* SSLCERTUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 192, (int)$index, (string)$value, 0);
        break;
      }


      case 193: { /* SSLCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 193, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLCertUsageFlags.", 0);
        }
        break;
      }


      case 194: { /* SSLCERTVERSION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 194, (int)$index, (string)$value, 0);
        break;
      }


      case 195: { /* SSLCERTSUBJECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 195, (int)$index, (string)$value, 0);
        break;
      }


      case 196: { /* SSLCERTENCODED */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 196, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 197: { /* SSLPROVIDER */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 197, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLProvider.", 0);
        }
        break;
      }


      case 198: { /* SSLSERVERCERTEFFECTIVEDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 198, (int)$index, (string)$value, 0);
        break;
      }


      case 199: { /* SSLSERVERCERTEXPIRATIONDATE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 199, (int)$index, (string)$value, 0);
        break;
      }


      case 200: { /* SSLSERVERCERTEXTENDEDKEYUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 200, (int)$index, (string)$value, 0);
        break;
      }


      case 201: { /* SSLSERVERCERTFINGERPRINT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 201, (int)$index, (string)$value, 0);
        break;
      }


      case 202: { /* SSLSERVERCERTFINGERPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 202, (int)$index, (string)$value, 0);
        break;
      }


      case 203: { /* SSLSERVERCERTFINGERPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 203, (int)$index, (string)$value, 0);
        break;
      }


      case 204: { /* SSLSERVERCERTISSUER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 204, (int)$index, (string)$value, 0);
        break;
      }


      case 205: { /* SSLSERVERCERTPRIVATEKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 205, (int)$index, (string)$value, 0);
        break;
      }


      case 206: { /* SSLSERVERCERTPRIVATEKEYAVAILABLE */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 206, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertPrivateKeyAvailable.", 0);
        }
        break;
      }


      case 207: { /* SSLSERVERCERTPRIVATEKEYCONTAINER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 207, (int)$index, (string)$value, 0);
        break;
      }


      case 208: { /* SSLSERVERCERTPUBLICKEY */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 208, (int)$index, (string)$value, 0);
        break;
      }


      case 209: { /* SSLSERVERCERTPUBLICKEYALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 209, (int)$index, (string)$value, 0);
        break;
      }


      case 210: { /* SSLSERVERCERTPUBLICKEYLENGTH */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 210, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertPublicKeyLength.", 0);
        }
        break;
      }


      case 211: { /* SSLSERVERCERTSERIALNUMBER */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 211, (int)$index, (string)$value, 0);
        break;
      }


      case 212: { /* SSLSERVERCERTSIGNATUREALGORITHM */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 212, (int)$index, (string)$value, 0);
        break;
      }


      case 213: { /* SSLSERVERCERTSTORE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 213, (int)$index, (string)$value, strlen((string)$value));
        break;
      }


      case 214: { /* SSLSERVERCERTSTOREPASSWORD */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 214, (int)$index, (string)$value, 0);
        break;
      }


      case 215: { /* SSLSERVERCERTSTORETYPE */
        if (is_string($value) || is_long($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 215, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertStoreType.", 0);
        }
        break;
      }


      case 216: { /* SSLSERVERCERTSUBJECTALTNAMES */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 216, (int)$index, (string)$value, 0);
        break;
      }


      case 217: { /* SSLSERVERCERTTHUMBPRINTMD5 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 217, (int)$index, (string)$value, 0);
        break;
      }


      case 218: { /* SSLSERVERCERTTHUMBPRINTSHA1 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 218, (int)$index, (string)$value, 0);
        break;
      }


      case 219: { /* SSLSERVERCERTTHUMBPRINTSHA256 */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 219, (int)$index, (string)$value, 0);
        break;
      }


      case 220: { /* SSLSERVERCERTUSAGE */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 220, (int)$index, (string)$value, 0);
        break;
      }


      case 221: { /* SSLSERVERCERTUSAGEFLAGS */
        if (is_string($value) || is_long($value) || is_bool($value)) {
            $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 221, (int)$index,
              $lib_cloudmail_office365->cast("void*", (int)$value), 0);
        } else {
            error_log("Invalid value specified for SSLServerCertUsageFlags.", 0);
        }
        break;
      }


      case 222: { /* SSLSERVERCERTVERSION */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 222, (int)$index, (string)$value, 0);
        break;
      }


      case 223: { /* SSLSERVERCERTSUBJECT */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 223, (int)$index, (string)$value, 0);
        break;
      }


      case 224: { /* SSLSERVERCERTENCODED */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 224, (int)$index, (string)$value, strlen((string)$value));
        break;
      }

      case 8001/*PID_RUNTIME_LICENSE*/: { /* RuntimeLicense */
        $retval = $lib_cloudmail_office365->CloudMail_Office365_Set($this->handle, 8001/*PID_RUNTIME_LICENSE*/, (int)$index, (string)$value, 0);
        break;
      }
      default: {
        error_log("Invalid property id", 0);
        break;
      }
    }
    return $retval;
  }
}


?>

