<?php
namespace Espo\Modules\ThreadedComments\Services;

use Espo\Core\Templates\Services\Base;
use Espo\Core\ORM\Entity as CoreEntity;
use Espo\Core\Utils\Json;
use Espo\Core\Utils\Util;
use Espo\ORM\Entity;
use Espo\ORM\EntityCollection;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Di;

class Comment extends Base implements
    Di\AclManagerAware,
    Di\ConfigAware,
    Di\FileManagerAware,
    Di\LanguageAware,
    Di\DateTimeAware,
    Di\NumberAware,
    Di\InjectableFactoryAware
{
    use Di\AclManagerSetter;
    use Di\ConfigSetter;
    use Di\FileManagerSetter;
    use Di\LanguageSetter;
    use Di\DateTimeSetter;
    use Di\NumberSetter;
    use Di\InjectableFactorySetter;

    public function findForEntity(string $scope, string $id, $searchParams = null): EntityCollection
    {
        $entity = $this->entityManager->getEntity($scope, $id);

        if (!$entity) {
            throw new NotFound();
        }

        if (!$this->aclManager->checkStream($this->user, $entity)) {
            throw new Forbidden();
        }

        $searchParams = $searchParams ?? $this->getSelectManagerFactory()->create('Comment')->getEmptySelectParams();

        $searchParams['whereClause'] = [
            'parentType' => $scope,
            'parentId' => $id,
            'commentReplyId' => null
        ];

        $searchParams['orderBy'] = 'createdAt';
        $searchParams['order'] = 'DESC';

        $collection = $this->repository->find($searchParams);

        foreach ($collection as $e) {
            $this->loadListAdditionalFields($e, $searchParams);
        }

        return $collection;
    }

    public function getCommentThreadTree(string $threadId): array
    {
        $thread = $this->entityManager->getEntity('Comment', $threadId);

        if (!$thread) {
            throw new NotFound();
        }

        if (!$this->aclManager->checkStream($this->user, $thread)) {
            throw new Forbidden();
        }

        $searchParams = $this->getSelectManagerFactory()->create('Comment')->getEmptySelectParams();

        $searchParams['whereClause'] = [
            'threadId' => $threadId
        ];

        $searchParams['orderBy'] = 'createdAt';
        $searchParams['order'] = 'ASC';

        $collection = $this->repository->find($searchParams);

        $list = [];

        foreach ($collection as $e) {
            $this->loadListAdditionalFields($e, $searchParams);
            $list[] = $e->getValueMap();
        }

        return $this->buildThreadTree($list, $threadId);
    }

    protected function buildThreadTree(array &$comments, $rootCommentId): array
    {
        $tree = [];

        foreach ($comments as $comment) {
            if ($comment->commentReplyId === $rootCommentId) {
                $comment->children = $this->buildThreadTree($comments, $comment->id);
                $tree[] = $comment;
            }
        }

        return $tree;
    }

    protected function loadListAdditionalFields(Entity $entity, $searchParams = null): void
    {
        $this->loadAdditionalFields($entity);

        $entity->set('hasChildren', $this->checkHasChildren($entity));
    }

    protected function checkHasChildren(Entity $entity): bool
    {
        $count = $this->entityManager->getRepository('Comment')->where([
            'commentReplyId' => $entity->id
        ])->count();

        return $count > 0;
    }

    protected function beforeCreateEntity(Entity $entity, $data)
    {
        if ($entity->get('commentReplyId')) {
            $replyTo = $this->entityManager->getEntity('Comment', $entity->get('commentReplyId'));

            if (!$replyTo) {
                throw new BadRequest("Reply-to comment doesn't exist.");
            }

            if (!$this->aclManager->checkStream($this->user, $replyTo)) {
                throw new Forbidden();
            }

            $entity->set('threadId', $replyTo->get('threadId') ?? $replyTo->id);
        }
    }
} 